// registro.js — Versión unificada (canjes centralizados)
// Depende del HTML con: #coinsModalOverlay, #carnetUsuario, #btnReclamarPremio, #login-modal
// y del endpoint /api/solicitudes.php que valida telefono+monto (PHP provisto)

(function(){
  /* =============================
   *  CONFIG / HELPERS
   * ============================= */
  const BE = (window.BACKEND || '/api/').replace(/\/?$/, '/');

  const isOk = (obj) => !!obj && (obj.ok === true || obj.success === true || obj.status === 'ok' || obj.created === true);

  const parseJSONSafe = async (res) => {
    const raw = await res.text();
    try { return { json: JSON.parse(raw), raw }; }
    catch { return { json: null, raw }; }
  };

  const showNeonAlert = (msg) => {
    if (typeof window.showNeonAlert === 'function') return window.showNeonAlert(msg);
    alert(msg);
  };

  // Overlay de bloqueo “Espere…”
  let _blocker;
  const showBlocker = (texto = 'Espere…') => {
    if (_blocker) { _blocker.querySelector('.blk-msg').textContent = texto; _blocker.style.display = 'flex'; return; }
    const div = document.createElement('div');
    div.className = 'blk-overlay';
    div.style.cssText = `position:fixed; inset:0; z-index:100000; background:rgba(0,0,0,.72); display:flex; align-items:center; justify-content:center; backdrop-filter: blur(2px);`;
    div.innerHTML = `
      <div style="background:#0b0d13; border:2px solid #00c8ff; color:#e9eef7; padding:14px 18px; border-radius:12px; box-shadow:0 0 16px rgba(0,200,255,.33); font-family: system-ui, 'Segoe UI', Roboto, Arial, sans-serif; display:flex; gap:10px; align-items:center;">
        <span class="spinner" style="width:18px;height:18px;border-radius:50%;border:2px solid #00c8ff;border-top-color:transparent; display:inline-block; animation:spin .8s linear infinite;"></span>
        <strong class="blk-msg">${texto}</strong>
      </div>
      <style>@keyframes spin { to { transform: rotate(360deg); } }</style>
    `;
    document.body.appendChild(div);
    _blocker = div;
  };
  const hideBlocker = () => { if (_blocker) _blocker.style.display = 'none'; };

  // Guarda una traza local de canjes
  const logCanjeLocal = (entry) => {
    try {
      const arr = JSON.parse(localStorage.getItem('solicitudesCanje') || '[]');
      arr.push(entry);
      localStorage.setItem('solicitudesCanje', JSON.stringify(arr));
      localStorage.setItem('ultimoCanje', JSON.stringify(entry));
    } catch {}
  };

  /* =============================
   *  REGISTRO DE USUARIO (mínimo, respeta tu HTML)
   * ============================= */
  document.addEventListener('DOMContentLoaded', () => {
    const btnAbrir = document.querySelector('.btn-registro');
    const modalRegistro = document.getElementById('modal-registro') || document.querySelector('.modal-registro');
    const btnCerrarRegistro = modalRegistro ? modalRegistro.querySelector('.cerrar-registro') : null;
    const btnEnviar = document.getElementById('enviar-registro');
    const errorMsg = modalRegistro ? modalRegistro.querySelector('.registro-error') : null;
    const modalExito = document.getElementById('modal-registro-exito') || document.querySelector('.modal-registro-exito');
    const btnCerrarExito = modalExito ? modalExito.querySelector('.cerrar-exito') : null;

    // Abrir/Cerrar registro
    if (btnAbrir && modalRegistro && errorMsg) {
      btnAbrir.addEventListener('click', () => { errorMsg.style.display = 'none'; modalRegistro.style.display = 'flex'; });
    }
    if (btnCerrarRegistro && modalRegistro && errorMsg) {
      btnCerrarRegistro.addEventListener('click', () => { modalRegistro.style.display = 'none'; errorMsg.style.display = 'none'; });
    }
    if (btnCerrarExito && modalExito) {
      btnCerrarExito.addEventListener('click', () => modalExito.style.display = 'none');
    }

    // Enviar registro (si tu HTML lo incluye)
    if (btnEnviar && errorMsg && modalRegistro && modalExito) {
      btnEnviar.addEventListener('click', async () => {
        const codigo = document.getElementById('codigo-operadora')?.value.trim();
        const telefono = document.getElementById('telefono')?.value.trim();
        const nombre = document.getElementById('nombre')?.value.trim();
        const telefonosValidos = ['0412','0414','0416','0424','0426','0422'];

        if (!nombre) { errorMsg.textContent = 'Por favor, ingresa tu nombre completo.'; errorMsg.style.display='block'; return; }
        if (!telefonosValidos.includes(codigo)) { errorMsg.textContent = 'Código de operadora inválido.'; errorMsg.style.display='block'; return; }
        if (!/^\d{7}$/.test(telefono || '')) { errorMsg.textContent = 'El teléfono debe tener 7 dígitos numéricos.'; errorMsg.style.display='block'; return; }

        const numeroCompleto = codigo + telefono;
        try {
          showBlocker('Registrando…');
          const res = await fetch(BE + 'register.php', {
            method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ nombre, telefono: numeroCompleto })
          });
          const { json, raw } = await parseJSONSafe(res);
          if (!res.ok || !json) throw new Error('Respuesta inválida del servidor: ' + raw);

          if (isOk(json)) {
            sessionStorage.setItem('loggedPhone', numeroCompleto);
            modalRegistro.style.display = 'none'; errorMsg.style.display = 'none';
            const telEl = document.getElementById('telefono'); if (telEl) telEl.value = '';
            const nomEl = document.getElementById('nombre'); if (nomEl) nomEl.value = '';
            try {
              if (typeof fetchUsuarioPorTelefono === 'function') {
                const u = await fetchUsuarioPorTelefono(numeroCompleto);
                if (typeof window.updateSessionUI === 'function') window.updateSessionUI(u);
              } else if (json.usuario && typeof window.updateSessionUI === 'function') {
                window.updateSessionUI(json.usuario);
              }
            } catch {}
            modalExito.style.display = 'flex';
          } else {
            const msg = json.error || json.message || 'No se pudo registrar.';
            errorMsg.textContent = msg; errorMsg.style.display = 'block';
          }
        } catch (e) {
          console.error('[Registro] error:', e);
          errorMsg.textContent = e.message || 'Error de red.'; errorMsg.style.display = 'block';
        } finally { hideBlocker(); }
      });
    }
  });

  /* =============================
   *  puntos: CARNET + CANJE
   * ============================= */
  const fetchUserByPhone = async (tel) => {
    if (!tel) return null;
    // Reutiliza fetch central si existe
    if (typeof window.fetchUsuarioPorTelefono === 'function') {
      try { return await window.fetchUsuarioPorTelefono(tel); } catch { /* fallback abajo */ }
    }
    try {
      const res = await fetch(BE + 'usuarios.php?telefono=' + encodeURIComponent(tel), { cache: 'no-store' });
      const { json } = await parseJSONSafe(res);
      const getRawUser = (d) => {
        if (!d) return null;
        if (d.usuario) return d.usuario;
        if (d.user) return d.user;
        if (d.data && !Array.isArray(d.data) && typeof d.data === 'object') return d.data;
        if (Array.isArray(d.data) && d.data.length) return d.data[0];
        if (Array.isArray(d) && d.length) return d[0];
        return null;
      };
      const raw = getRawUser(json);
      if (!raw) return null;
      return {
        id: raw.id || null,
        nombre: raw.nombre || raw.name || '-',
        telefono: raw.telefono || raw.phone || tel,
        puntos: parseInt(raw.puntos ?? raw.coins ?? raw.puntos ?? 0) || 0,
        maxpuntos: parseInt(raw.maxpuntos ?? raw.maxCoins ?? raw.puntos ?? 0) || 0,
        avatarURL: raw.avatarURL || raw.avatarUrl || raw.avatar || ''
      };
    } catch (e) {
      console.warn('[fetchUserByPhone] error:', e);
      return null;
    }
  };

  // Envío robusto de solicitud de canje (prueba JSON y luego FORM)
  async function enviarSolicitudCanje(payload, signal){
    // 1) intento JSON
    try {
      const res = await fetch(BE + 'solicitudes.php', {
        method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(payload), cache: 'no-store', signal
      });
      const { json, raw } = await parseJSONSafe(res);
      if (res.ok && json && isOk(json)) return { ok: true, data: json };
      console.warn('[canje][JSON] status=', res.status, json || raw);
      if (json && (json.error || json.message)) return { ok:false, data: json };
    } catch (e) { if (e?.name !== 'AbortError') console.warn('[canje][JSON] error:', e); }

    // 2) intento x-www-form-urlencoded
    try {
      const form = new URLSearchParams(); Object.entries(payload).forEach(([k,v]) => form.append(k, String(v ?? '')));
      const res2 = await fetch(BE + 'solicitudes.php', {
        method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' }, body: form, cache: 'no-store', signal
      });
      const { json: json2, raw: raw2 } = await parseJSONSafe(res2);
      if (res2.ok && json2 && isOk(json2)) return { ok: true, data: json2 };
      const msg = (json2 && (json2.error || json2.message)) || `Error HTTP ${res2.status}: ${raw2}`;
      return { ok:false, data:{ error: msg } };
    } catch (e2) { if (e2?.name !== 'AbortError') console.warn('[canje][FORM] error:', e2); return { ok:false, data:{ error:'Error de red' } }; }
  }
  // expone por si algo externo quiere reusar
  window.enviarSolicitudCanje = enviarSolicitudCanje;

  // Muestra el carnet y configura el botón de canje
  window.mostrarCarnetUsuario = async function (tel) {
    const overlay = document.getElementById('coinsModalOverlay');
    if (!tel || !overlay) return;

    // 1) Obtener usuario
    let user = null; try { user = await fetchUserByPhone(tel); } catch(e){ console.error('[Carnet] user:', e); }

    // 2) Mostrar overlay + tarjeta
    overlay.removeAttribute('hidden'); overlay.style.display = 'flex';
    const carnetCard = overlay.querySelector('.carnet#carnetUsuario') || document.getElementById('carnetUsuario');
    if (carnetCard) carnetCard.style.display = 'flex';

    // 3) Selectores
    const imgAvatar   = carnetCard ? carnetCard.querySelector('img.avatar') : null;
    const infoName    = carnetCard ? carnetCard.querySelector('.info .name span') : null;
    const infoCoins   = carnetCard ? carnetCard.querySelector('.info .coins span') : null;
    const infoLevel   = carnetCard ? carnetCard.querySelector('.info .categoria span') : null;
    const premiosP    = carnetCard ? carnetCard.querySelector('#carnet-premios') : null;
    const btnReclamar = document.getElementById('btnReclamarPremio');

    // 4) Placeholders si no hay user
    if (!user) {
      infoName  && (infoName.textContent  = '-');
      infoCoins && (infoCoins.textContent = '0');
      infoLevel && (infoLevel.textContent = '👶 Borrachin sin Nivel');
      premiosP  && (premiosP.textContent  = 'No registrado.');
      btnReclamar && (btnReclamar.disabled = true);
      return;
    }

    // 5) Pintar datos
    imgAvatar && (imgAvatar.src = user.avatarURL || 'avatar.jpg');
    infoName  && (infoName.textContent  = user.nombre || '-');
    infoCoins && (infoCoins.textContent = String(user.puntos || 0));

    const nivelCoins = parseInt(user.maxpuntos || user.puntos || 0);
    let nivel = '👶 Borrachin sin Nivel';
    if (nivelCoins >= 500) nivel = '🍻 Tan Bello VIP';
    else if (nivelCoins >= 300) nivel = '🍺 Borrachín De Nivel 3';
    else if (nivelCoins >= 100) nivel = '🥉 Borrachín Plateado';
    infoLevel && (infoLevel.textContent = nivel);

    // Saldo actual
    const c = parseInt(user.puntos || 0);

    // Texto del premio (tiers)
    let premiosText = 'Muy duro, aún no tienes suficientes puntos!';
    if      (c >= 500) premiosText = '🍺 10 Producto 1 gratis!';
    else if (c >= 400) premiosText = '🍺 8 Producto 1 gratis!';
    else if (c >= 300) premiosText = '🍺 6 Producto 1 gratis!';
    else if (c >= 200) premiosText = '🍺 4 Producto 1 gratis!';
    else if (c >= 100) premiosText = '🍺 2 Producto 1 gratis!';
    premiosP && (premiosP.textContent  = premiosText);

    // Premio automático → fija el monto exacto en el botón
    let bestCost = 0;
    if      (c >= 500) bestCost = 500;
    else if (c >= 400) bestCost = 400;
    else if (c >= 300) bestCost = 300;
    else if (c >= 200) bestCost = 200;
    else if (c >= 100) bestCost = 100;

    if (btnReclamar) {
      if (bestCost > 0) { btnReclamar.disabled = false; btnReclamar.dataset.monto = String(bestCost); }
      else { btnReclamar.disabled = true; btnReclamar.removeAttribute('data-monto'); }
    }

    // Limpieza de posibles qty residuales
    document.querySelectorAll('#carnetUsuario [data-qty], #carnetUsuario .qty')
      .forEach(el => { try { el.value = 0; el.dataset.qty = 0; } catch {} });
  };

  // === Cerrar carnet (botón X y click fuera) ===
  document.addEventListener('click', (e) => {
    if (e.target.closest('.carnet-close')) {
      const overlay = document.getElementById('coinsModalOverlay');
      if (overlay) { overlay.style.display = 'none'; overlay.setAttribute('hidden', ''); }
    }
  });
  document.getElementById('coinsModalOverlay')?.addEventListener('click', (e) => {
    const card = document.getElementById('carnetUsuario');
    if (card && !card.contains(e.target)) { e.currentTarget.style.display = 'none'; e.currentTarget.setAttribute('hidden', ''); }
  });

  /* =============================
   *  RECLAMAR PREMIO (canje) — CENTRALIZADO AQUÍ
   * ============================= */
  let reclamoCtrl = null;
  async function reclamarPremio(montoForzado = null, botonRef = null) {
    const tel = sessionStorage.getItem('loggedPhone');
    if (!tel) { const lm = document.getElementById('login-modal'); if (lm) lm.style.display = 'flex'; return; }

    const user = await fetchUserByPhone(tel);
    const saldo = Number(user?.puntos || 0);

    // Tomamos el monto del botón (seteado en mostrarCarnetUsuario), o el forzado
    const btn = document.getElementById('btnReclamarPremio');
    let monto = Number(montoForzado ?? (btn?.dataset?.monto || 0)) || 0;
    if (!monto || monto <= 0) { showNeonAlert('⚠️ Tu saldo no alcanza para ningún premio.'); return; }
    if (monto > saldo) { showNeonAlert(`❌ Saldo insuficiente: tienes ${saldo} y quieres canjear ${monto}.`); return; }

    // Guardias de UI
    if (botonRef) { if (botonRef.dataset.busy === '1') return; botonRef.dataset.busy = '1'; botonRef.disabled = true; }
    if (reclamoCtrl) try { reclamoCtrl.abort(); } catch {}
    reclamoCtrl = new AbortController();

    try {
      showBlocker('Enviando solicitud de canje…');
      const payload = { accion:'crear', tipo:'canjear', monto, telefono: tel, nombre: user?.nombre || '' };
      const { ok, data } = await enviarSolicitudCanje(payload, reclamoCtrl.signal);

      if (ok && data && (data.success === true || data.created === true)) {
        const msg = String(data.message || '').toLowerCase();
        if (msg.includes('ya registrada')) showNeonAlert('ℹ️ Ya tenías una solicitud pendiente igual.');
        else showNeonAlert('✅ Solicitud enviada. Espera aprobación del admin.');
        logCanjeLocal({ fecha: Date.now(), telefono: tel, monto, code: data.code || null });
        // refresca carnet
        if (typeof window.mostrarCarnetUsuario === 'function') window.mostrarCarnetUsuario(tel);
      } else {
        const emsg = (data && (data.error || data.message)) || 'No se pudo crear la solicitud';
        showNeonAlert('❌ ' + emsg);
      }
    } catch (e) {
      if (e?.name !== 'AbortError') showNeonAlert('❌ Error de red creando solicitud');
    } finally {
      hideBlocker();
      if (botonRef) { botonRef.dataset.busy = '0'; botonRef.disabled = false; }
    }
  }
  window.reclamarPremio = reclamarPremio;

  // Delegado único para el botón de canje
  document.addEventListener('click', (e) => {
    const btn = e.target.closest('#btnReclamarPremio');
    if (!btn) return;
    e.preventDefault();
    const monto = Number(btn.dataset.monto || 0) || null;
    reclamarPremio(monto, btn);
  }, true);

})();
