<?php
// /api/solicitudes.php
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
ini_set('display_errors', '0');

require_once __DIR__ . '/../backend/db.php'; // ajusta la ruta si tu db.php vive en otro lado

function read_input() {
  $raw = file_get_contents('php://input');
  $json = json_decode($raw, true);
  if (is_array($json)) return $json;

  // fallback a x-www-form-urlencoded
  if (!empty($_POST)) return $_POST;

  // también aceptar querystring en emergencia
  if (!empty($_GET)) return $_GET;

  return [];
}
function json_out($arr, $code = 200) {
  http_response_code($code);
  echo json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

try {
  $in = read_input();

  // Normalización de campos esperados desde el front
  $accion   = strtolower(trim($in['accion']   ?? 'crear'));
  $tipo     = strtolower(trim($in['tipo']     ?? 'canjear'));
  $monto    = (int)($in['monto']              ?? 0);
  $telefono = preg_replace('/\D+/', '', strval($in['telefono'] ?? ''));
  $nombre   = trim($in['nombre'] ?? '');

  if ($accion !== 'crear') {
    json_out(['success' => false, 'error' => 'Acción inválida'], 400);
  }
  if ($tipo !== 'canjear') {
    json_out(['success' => false, 'error' => 'Tipo inválido'], 400);
  }
  if ($monto <= 0) {
    json_out(['success' => false, 'error' => 'Monto inválido'], 400);
  }
  if (strlen($telefono) < 10) {
    json_out(['success' => false, 'error' => 'Teléfono inválido'], 400);
  }

  // Buscar usuario por teléfono
  $stmt = $mysqli->prepare("SELECT id, nombre, telefono, cumanaCoins FROM usuarios WHERE telefono=? LIMIT 1");
  if (!$stmt) json_out(['success'=>false,'error'=>'DB error (prep user)'], 500);
  $stmt->bind_param("s", $telefono);
  $stmt->execute();
  $u = $stmt->get_result()->fetch_assoc();
  $stmt->close();

  if (!$u) {
    json_out(['success'=>false,'error'=>'Usuario no encontrado con ese teléfono'], 404);
  }

  $userId = (int)$u['id'];

  // (Opcional pero recomendado) Validación de saldo en servidor
  $saldo = (int)($u['cumanaCoins'] ?? 0);
  if ($saldo < $monto) {
    json_out(['success'=>false,'error'=>"Saldo insuficiente: saldo=$saldo, intento=$monto"], 400);
  }

  // Evitar duplicado pendiente igual (mismo usuario y mismo monto)
  $stmt = $mysqli->prepare("SELECT id FROM solicitudes 
                            WHERE userId=? AND tipo='canjear' AND estado='pendiente' AND monto=? 
                            ORDER BY id DESC LIMIT 1");
  if (!$stmt) json_out(['success'=>false,'error'=>'DB error (prep dup)'], 500);
  $stmt->bind_param("ii", $userId, $monto);
  $stmt->execute();
  $dup = $stmt->get_result()->fetch_assoc();
  $stmt->close();

  if ($dup) {
    json_out([
      'success' => true,
      'message' => 'Solicitud ya registrada (pendiente).',
      'duplicated' => true,
      'id' => (int)$dup['id']
    ], 200);
  }

  // Generar código de canje
  try {
    $code = 'CANJ-' . bin2hex(random_bytes(4));
  } catch (Throwable $e) {
    $code = 'CANJ-' . strtoupper(substr(md5(uniqid('', true)), 0, 8));
  }

  // Insertar solicitud
  $stmt = $mysqli->prepare(
    "INSERT INTO solicitudes (userId, tipo, monto, estado, canje_code, notas, createdAt) 
     VALUES (?, 'canjear', ?, 'pendiente', ?, '', NOW())"
  );
  if (!$stmt) json_out(['success'=>false,'error'=>'DB error (prep insert)'], 500);
  $stmt->bind_param("iis", $userId, $monto, $code);
  $ok = $stmt->execute();
  $newId = $stmt->insert_id;
  $stmt->close();

  if (!$ok) {
    json_out(['success'=>false,'error'=>'No se pudo crear la solicitud'], 500);
  }

  // Nota: NO descontamos coins aquí; se descuenta cuando el admin aprueba
  json_out([
    'success' => true,
    'created' => true,
    'id'      => (int)$newId,
    'code'    => $code,
    'status'  => 'ok',
    'message' => 'Solicitud creada'
  ], 200);

} catch (Throwable $e) {
  json_out(['success'=>false, 'error'=>'Excepción: '.$e->getMessage()], 500);
}
