<?php
// backend/_upload_webp_only.php
declare(strict_types=1);

function slugify_file(string $name): string {
  $base = preg_replace('/[^a-z0-9._-]+/i','-', pathinfo($name, PATHINFO_FILENAME)) ?: 'img';
  return preg_replace('/-+/', '-', strtolower($base));
}

/**
 * Sube una imagen SOLO si es .webp real (MIME y extensión).
 * Si no es .webp, elimina el tmp y lanza Exception.
 *
 * @param array  $file  $_FILES['campo']
 * @param string $destDirRutaAbsoluta  /var/www/html/imgproductos/xxx
 * @return array [rel => '/imgproductos/xxx/nombre.webp', abs => '/var/www/html/imgproductos/xxx/nombre.webp']
 */
function upload_only_webp(array $file, string $destDirRutaAbsoluta): array {
  if (empty($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
    throw new RuntimeException('No se cargó archivo.');
  }
  $tmp  = $file['tmp_name'];
  $ext  = strtolower(pathinfo($file['name'] ?? 'img.webp', PATHINFO_EXTENSION));

  // Validar MIME real
  $fi   = new finfo(FILEINFO_MIME_TYPE);
  $mime = $fi->file($tmp) ?: '';

  if ($ext !== 'webp' || $mime !== 'image/webp') {
    @unlink($tmp); // quitarla
    throw new RuntimeException('Solo se aceptan imágenes .webp');
  }

  if (!is_dir($destDirRutaAbsoluta)) {
    if (!mkdir($destDirRutaAbsoluta, 0775, true)) {
      @unlink($tmp);
      throw new RuntimeException('No se pudo crear carpeta destino.');
    }
  }

  $base = slugify_file($file['name']) . '.webp';
  $abs  = rtrim($destDirRutaAbsoluta, '/').'/'.$base;

  if (!move_uploaded_file($tmp, $abs)) {
    @unlink($tmp);
    throw new RuntimeException('No se pudo guardar la imagen.');
  }
  @chmod($abs, 0644);

  // Derivar ruta pública relativa
  $docroot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? dirname(__DIR__,1), '/');
  $rel = str_replace($docroot, '', $abs);
  if ($rel === $abs) {
    // Fallback si el docroot no calza:
    $rel = '/'.ltrim(basename(dirname($destDirRutaAbsoluta)).'/'.$base, '/');
  }
  return ['rel' => $rel, 'abs' => $abs];
}
