/* ===========================
   chat.js (solo lógica JS)
   =========================== */
(function () {
  'use strict';

  // -------- BASE DEL BACKEND --------
  window.BACKEND = window.BACKEND || '/api/';
  const BE = String(window.BACKEND || '/api/').replace(/\/?$/, '/');
  // Base exclusiva para subida del presidente
const UPLOAD_BE = String(window.BACKEND_ADMIN || '/backend/').replace(/\/?$/, '/');


  // -------- UTILIDADES --------
  function $(sel, root = document) { return root.querySelector(sel); }
  function $all(sel, root = document) { return Array.from(root.querySelectorAll(sel)); }
  function showToast(msg) {
    // usa tu #neon-alert si existe; si no, alert()
    const el = document.getElementById('neon-alert');
    if (!el) { alert(msg); return; }
    el.textContent = msg;
    el.style.display = 'flex';
    el.classList.remove('show'); void el.offsetWidth;
    el.classList.add('show');
    el.addEventListener('animationend', function h() {
      el.classList.remove('show'); el.style.display = 'none'; el.removeEventListener('animationend', h);
    });
  }
  const escapeHtml = (s) => String(s).replace(/[&<>"']/g, m => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[m]));
  const isLogged = () => !!(sessionStorage.getItem('loggedPhone') || (window.currentUser && window.currentUser.telefono));
  const getMyPhone = () => sessionStorage.getItem('loggedPhone') || (window.currentUser && window.currentUser.telefono) || '';
  const getMyName  = () => (window.currentUser && window.currentUser.nombre) || 'Anon';

  // -------- NODOS --------
  const overlay   = document.getElementById('chat-overlay');
  const box       = overlay?.querySelector('.chat-box');
  const list      = document.getElementById('chat-messages');
  const attachBar = document.getElementById('chat-attach-bar');
  const emojiBtn  = document.getElementById('chat-emoji-btn');
  const fileInput = document.getElementById('chat-file-input');
  const inputEl   = document.getElementById('chat-message-input');
  const sendBtn   = document.getElementById('chat-send');
  const closeBtn  = document.getElementById('chat-close');
  const toggleBtn = document.getElementById('chat-toggle'); // opcional

  // --- Visibilidad del botón chat-toggle según sesión ---
  function refreshChatToggleVisibility() {
    const chatBtn = document.getElementById('chat-toggle');
    if (!chatBtn) return;
    const isLogged = !!(
      sessionStorage.getItem('loggedPhone') ||
      (window.currentUser && window.currentUser.telefono)
    );
    chatBtn.style.display = isLogged ? 'inline-flex' : 'none';
  }
  refreshChatToggleVisibility();
// === Mostrar el botón del chat inmediatamente después del login (sin refresh) ===
(function instantChatToggle(){
  // 1) Detecta cuando se setea sessionStorage.loggedPhone en ESTA MISMA pestaña
  const origSetItem = sessionStorage.setItem;
  sessionStorage.setItem = function(key, value) {
    origSetItem.apply(this, arguments);
    if (key === 'loggedPhone') {
      // Pequeño delay para que termine la lógica de login
      setTimeout(() => {
        refreshChatToggleVisibility();
      }, 0);
    }
  };

  // 2) Si tu login asigna window.currentUser, detecta cambios y refresca
  try {
    let _cu = window.currentUser;
    Object.defineProperty(window, 'currentUser', {
      configurable: true,
      get(){ return _cu; },
      set(v){ _cu = v; refreshChatToggleVisibility(); return v; }
    });
  } catch {}

  // 3) Helper opcional por si quieres llamarlo desde tu flujo de login
  //    window.notifyLogin(usuario) -> mostrará el botón al instante
  window.notifyLogin = function(user){
    if (user) window.currentUser = user;
    refreshChatToggleVisibility();
  };
})();

  // si otra parte de la app actualiza la sesión, intenta engancharte:
  (function hookSessionUI(){
    const prev = window.updateSessionUI;
    window.updateSessionUI = function(user){
      if (typeof prev === 'function') prev(user);
      refreshChatToggleVisibility();
    };
  })();

  // por si cambia sessionStorage en otra pestaña
  window.addEventListener('storage', (e) => {
    if (e.key === 'loggedPhone') refreshChatToggleVisibility();
  });

  // -------- STATE --------
  let pollTimer = null;
  let lastId = null;
  const seenKeys = new Set();
  const selectedFiles = [];
  const msgIndex = {};        // id -> {nombre,texto}
  let replyCtx   = null;      // { id, nombre, texto }
  let whisperCtx = null;      // { telefono, nombre }
  let emojiPop   = null;
  let chatPollTimer = null;   // si usas las funciones extendidas más abajo

  // -------- HELPERS --------
  function isAtBottom(el, tol = 12) {
    if (!el) return true;
    return (el.scrollHeight - el.clientHeight - el.scrollTop) <= tol;
  }
  function hash32(str) {
    let h = 2166136261 >>> 0;
    for (let i = 0; i < str.length; i++) { h ^= str.charCodeAt(i); h = Math.imul(h, 16777619) >>> 0; }
    return (h >>> 0).toString(16);
  }
  function stableKey(m) {
    const id = m.id ?? m.ID ?? m.mid ?? m.msg_id;
    if (id != null && String(id) !== '') return 'id:' + String(id);
    const t = m.fecha ? (Date.parse(m.fecha) || 0) : 0;
    const tel = m.telefono || '';
    const txt = (m.texto || '').slice(0, 200);
    return `sig:${t}|${tel}|${txt.length}|${hash32(txt)}`;
  }
  const getPhone   = m => m.telefono || m.phone || m.tel || m.from || '';
  const getToPhone = m => m.para_telefono || m.para || m.to || m.destino || '';

  function setChatMode(mode) {
    if (!box) return;
    box.classList.remove('mode-reply', 'mode-whisper');
    if (mode) box.classList.add('mode-' + mode);
  }
  // ---- IDs del backend / Reply / Susurros ----
  function getBackendId(m){
    return m.id ?? m.ID ?? m.mid ?? m.msg_id ?? null;
  }
  function getReplyId(m){
    return m.reply_to ?? m.replyTo ?? m.responde_a ?? m.reply ?? null;
  }
  function getReplyName(m){
    return m.reply_nombre ?? m.replyName ?? m.reply_user ?? null;
  }
  function getReplyExcerpt(m){
    return m.reply_excerpt ?? m.replyText ?? m.reply_texto ?? null;
  }
  function isWhisperMsg(m){
    const tipo = (m.tipo || '').toString().toLowerCase();
    return !!(m.privado || m.private || m.is_private || m.susurro || m.whisper || tipo === 'whisper');
  }
  function getRecipientPhone(m){
    return m.para_telefono ?? m.paraTelefono ?? m.para ?? m.to ?? m.destino ?? null;
  }

  // -------- CONTEXT BAR (Responder / Susurro) --------
  function ensureContextBar() {
    let bar = document.getElementById('chat-context-bar');
    if (bar) return bar;
    bar = document.createElement('div');
    bar.id = 'chat-context-bar';
    const inputRow = box?.querySelector('.chat-input');
    if (inputRow && box) box.insertBefore(bar, inputRow); else document.body.appendChild(bar);
    return bar;
  }
  function clearNode(node) { while (node && node.firstChild) node.removeChild(node.firstChild); }
  function renderContextBar() {
    const bar = ensureContextBar();
    clearNode(bar);
    bar.classList.remove('whisper');

    if (!replyCtx && !whisperCtx) {
      bar.style.display = 'none';
      setChatMode(null);
      return;
    }
    if (whisperCtx) bar.classList.add('whisper');

    const wrap = document.createElement('div'); wrap.className = 'ctx';
    const color = document.createElement('div'); color.className = 'ctx-color';
    const mid = document.createElement('div'); mid.className = 'ctx-mid';
    const title = document.createElement('div'); title.className = 'ctx-title';
    const excerpt = document.createElement('div'); excerpt.className = 'ctx-excerpt';
    const actions = document.createElement('div'); actions.className = 'ctx-actions';
    const cancel = document.createElement('button'); cancel.type = 'button'; cancel.className = 'chat-action'; cancel.textContent = 'Cancelar';

    const isWh = Boolean(whisperCtx);
    title.textContent = isWh ? `🤫 Susurro a ${whisperCtx?.nombre || ''}` : `↩ Respondiendo a ${replyCtx?.nombre || ''}`;
    excerpt.textContent = (replyCtx?.texto || '').slice(0, 180) || '…';

    cancel.addEventListener('click', () => { replyCtx = null; whisperCtx = null; renderContextBar(); });

    actions.appendChild(cancel);
    mid.appendChild(title); mid.appendChild(excerpt);
    wrap.appendChild(color); wrap.appendChild(mid); wrap.appendChild(actions);
    bar.appendChild(wrap);
    bar.style.display = 'block';

    if (whisperCtx) setChatMode('whisper');
    else if (replyCtx) setChatMode('reply');
    else setChatMode(null);
  }

  // -------- EMOJI PICKER (REMOVIDO) --------
  // Eliminado a petición. Si el botón existe en el DOM, lo ocultamos/removemos.
  if (typeof emojiBtn !== 'undefined' && emojiBtn) {
    try { emojiBtn.remove(); } catch { emojiBtn.style.display = 'none'; }
  }

  // -------- ATTACH BAR --------
  function renderAttachBar() {
    if (!attachBar) return;
    clearNode(attachBar);
    if (!selectedFiles.length) { attachBar.hidden = true; return; }
    selectedFiles.forEach((f, i) => {
      const chip = document.createElement('div'); chip.className = 'attach-chip';
      const name = document.createElement('span'); name.textContent = `📎 ${f.name}`;
      const x = document.createElement('button'); x.type = 'button'; x.setAttribute('aria-label', 'Quitar'); x.textContent = '×';
      x.addEventListener('click', () => { selectedFiles.splice(i, 1); renderAttachBar(); });
      chip.appendChild(name); chip.appendChild(x);
      attachBar.appendChild(chip);
    });
    attachBar.hidden = false;
  }
  if (fileInput) {
    fileInput.setAttribute('accept', 'image/*,video/*');
    fileInput.addEventListener('change', () => {
      const files = Array.from(fileInput.files || []);
      files.forEach(f => {
        const isMedia = /^image\//.test(f.type) || /^video\//.test(f.type);
        if (!isMedia) { showToast('Solo se permiten fotos y videos'); return; }
        if (f.size > 15 * 1024 * 1024) { showToast(`"${f.name}" supera 15 MB`); return; }
        selectedFiles.push(f);
      });
      renderAttachBar();
      fileInput.value = '';
    });
  }

  // -------- VIEWER (lightbox) --------
  function ensureViewer() {
    let v = document.getElementById('chat-viewer');
    if (v) return v;
    v = document.createElement('div'); v.id = 'chat-viewer';
    const close = document.createElement('button'); close.id = 'chat-viewer-close'; close.textContent = '✕';
    const body = document.createElement('div'); body.id = 'chat-viewer-body';
    v.appendChild(close); v.appendChild(body);
    document.body.appendChild(v);
    v.addEventListener('click', (e) => { if (e.target === v || e.target === close) v.classList.remove('show'); });
    return v;
  }
  function openViewer(src, type) {
    const v = ensureViewer();
    const body = v.querySelector('#chat-viewer-body');
    clearNode(body);
    let el;
    if ((type || '').startsWith('video/')) {
      el = document.createElement('video'); el.src = src; el.controls = true; el.autoplay = true;
    } else {
      el = document.createElement('img'); el.src = src; el.alt = 'adjunto';
    }
    el.id = 'chat-viewer-content';
    body.appendChild(el);
    v.classList.add('show');
  }
  // Ampliar la foto del presidente al tocarla
  (function () {
    const img = document.getElementById('chat-president-avatar');
    if (!img) return;
    img.addEventListener('click', () => {
      try {
        // reutiliza tu lightbox del chat
        openViewer(img.src, 'image/*');
      } catch (e) {
        console.warn('No se pudo abrir el visor:', e);
      }
    });
  })();

  // -------- NUEVOS MENSAJES --------
  function ensureNewMsgButton() {
    let btn = document.getElementById('jump-to-latest');
    if (btn) return btn;
    btn = document.createElement('button');
    btn.id = 'jump-to-latest';
    btn.type = 'button';
    btn.textContent = 'Nuevos mensajes ↓';
    btn.addEventListener('click', () => {
      list.scrollTop = list.scrollHeight;
      btn.style.display = 'none';
    });
    (box || document.body).appendChild(btn);
    list?.addEventListener('scroll', () => { if (isAtBottom(list)) btn.style.display = 'none'; });
    return btn;
  }

  // -------- RENDER MENSAJE --------
  function renderMessage(m){
    if (!list) return;

    // clave estable para evitar duplicados
    const k = stableKey(m);
    if (seenKeys.has(k)) return;
    seenKeys.add(k);

    const myTel   = getMyPhone();
    const backendId = getBackendId(m);
    const esMio   = myTel && String(myTel) === String(m.telefono || '');
    const isWh    = isWhisperMsg(m);
    const toPhone = getRecipientPhone(m);

    // Privacidad: solo emisor o destinatario ven el susurro
    if (isWh && !(esMio || (toPhone && String(toPhone) === String(myTel)))) return;

    // Usamos SIEMPRE el id real del backend para indexar y responder
    const msgId = backendId != null ? String(backendId) : ('tmp:' + k);

    const row = document.createElement('div');
    row.className = 'chat-row';
    row.dataset.msgId = msgId;

    const bubble = document.createElement('div');
    bubble.className = 'chat-msg' + (esMio ? ' own' : '') + (isWh ? ' whisper' : '');
    row.appendChild(bubble);

    if (isWh){
      const meta = document.createElement('div');
      meta.className = 'chat-meta-badge whisper';
      meta.textContent = '🤫 Susurro';
      bubble.appendChild(meta);
    }

    // ----- CITA (reply) -----
    const refId = getReplyId(m);
    if (refId != null){
      const ref = (msgIndex[String(refId)] || {});
      const refName = getReplyName(m) || ref.nombre || 'Mensaje';
      const refText = getReplyExcerpt(m) || ref.texto || '…';
      const refBox  = document.createElement('div'); refBox.className = 'chat-reply';
      const who     = document.createElement('span'); who.className='who'; who.textContent = refName + ' ';
      const txt     = document.createElement('div');  txt.className='txt'; txt.textContent = String(refText).slice(0,150);
      refBox.appendChild(who); refBox.appendChild(txt);
      bubble.appendChild(refBox);
    }

    // Línea principal
    const line = document.createElement('div');
    const who  = document.createElement('strong'); who.textContent = (m.nombre || 'Anon') + ': ';
    const txt  = document.createElement('span');   txt.textContent = m.texto || '';
    line.appendChild(who); line.appendChild(txt);
    bubble.appendChild(line);

    // Adjuntos
    const adj = (m.adjuntos || m.attachments || m.files) || [];
    if (Array.isArray(adj) && adj.length){
      const wrap = document.createElement('div'); wrap.className = 'chat-attachments';
      adj.forEach(a0=>{
        const a   = (typeof a0 === 'string') ? { url:a0 } : (a0 || {});
        const url = a.url || a.path || '';
        const mime= (a.tipo || a.type || a.mime || '').toLowerCase();
        let el;
        if (mime.startsWith('image/')){
          el = document.createElement('img'); el.src=url; el.alt=a.nombre||'imagen'; el.loading='lazy'; el.className='chat-attachment-thumb';
          el.addEventListener('click', ()=>openViewer(url, mime));
        } else if (mime.startsWith('video/')){
          el = document.createElement('video'); el.src=url; el.controls=true; el.className='chat-attachment-video';
          el.addEventListener('click', (e)=>{ if (e.target === el) openViewer(url, mime); });
        } else if (url){
          const l = document.createElement('a'); l.href=url; l.target='_blank'; l.rel='noopener'; l.textContent=a.nombre||url.split('/').pop(); l.className='chat-attachment-link';
          el = l;
        }
        if (el) wrap.appendChild(el);
      });
      bubble.appendChild(wrap);
    }

    // Hora
    const time = document.createElement('div'); time.className='chat-time';
    const when = m.fecha ? new Date(m.fecha) : new Date();
    try { time.textContent = when.toLocaleString('es-VE', { hour12:false }); }
    catch { time.textContent = new Date().toLocaleString(); }
    bubble.appendChild(time);

    // Acciones
    const acts = document.createElement('div'); acts.className='chat-actions';

    const bRep = document.createElement('button');
    bRep.type = 'button';
    bRep.className = 'chat-action is-reply';
    bRep.textContent = '↩ Responder';
    bRep.addEventListener('click', ()=>{
      // MUY IMPORTANTE: guardamos el ID REAL del backend
      const backendIdForReply = getBackendId(m);
      replyCtx   = { backendId: backendIdForReply, nombre: m.nombre || 'Usuario', texto: m.texto || '' };
      whisperCtx = null;
      renderContextBar();
    });
    acts.appendChild(bRep);

    if (!esMio){
      const bWh = document.createElement('button');
      bWh.type = 'button';
      bWh.className = 'chat-action is-whisper';
      bWh.textContent = '🤫 Susurrar';
      bWh.addEventListener('click', ()=>{
        const phone = getPhone(m) || getRecipientPhone(m) || prompt('Número del usuario para susurrar:');
        if (!phone) return;
        whisperCtx = { telefono: phone, nombre: m.nombre || 'Usuario' };
        // Mostramos también una pre-cita tipo WhatsApp
        const backendIdForReply = getBackendId(m);
        replyCtx = { backendId: backendIdForReply, nombre: m.nombre || 'Usuario', texto: m.texto || '' };
        renderContextBar();
      });
      acts.appendChild(bWh);
    }
    bubble.appendChild(acts);

    // Indexar para futuras citas (solo si hay ID real)
    if (backendId != null && !msgIndex[String(backendId)]){
      msgIndex[String(backendId)] = { nombre: m.nombre || 'Usuario', texto: m.texto || '' };
    }

    // DOM
    const estabaAbajo = isAtBottom(list);
    list.appendChild(row);
    if (estabaAbajo) list.scrollTop = list.scrollHeight;
    else if (!esMio) ensureNewMsgButton().style.display = 'inline-flex';

    // lastId (solo con IDs numéricos reales)
    if (backendId != null){
      const n = Number(backendId);
      if (!Number.isNaN(n)) lastId = (lastId == null) ? n : Math.max(lastId, n);
    }
  }

  // -------- CARGA / POLLING --------
  async function cargarMensajes({ full = false } = {}) {
    if (!list) return;
    try {
      const tel = getMyPhone();
      const url = new URL(BE + 'chat.php', window.location.origin);
      url.searchParams.set('_', Date.now().toString());        // rompe caché
      if (tel) url.searchParams.set('telefono', tel);          // ← MUY IMPORTANTE para susurros
      if (!full && lastId != null) url.searchParams.set('since_id', String(lastId));
      const r = await fetch(url.toString(), { cache: 'no-store' });

      const ct = r.headers.get('content-type') || '';
      const data = ct.includes('application/json')
        ? await r.json().catch(() => [])
        : JSON.parse(await r.text().catch(() => '[]'));
      if (!Array.isArray(data)) return;

      if (full) {
        list.innerHTML = '';
        Object.keys(msgIndex).forEach(k => delete msgIndex[k]);
        seenKeys.clear(); lastId = null;
      }

      // index previo para replies usando ID REAL del backend
      data.forEach(m => {
        const id = getBackendId(m);
        if (id != null && !msgIndex[String(id)]) {
          msgIndex[String(id)] = { nombre: m.nombre || 'Usuario', texto: m.texto || '' };
        }
      });
      data.forEach(renderMessage);

      if (full) list.scrollTop = list.scrollHeight;
    } catch (e) {
      console.warn('[chat] cargarMensajes error:', e);
    }
  }
  function startPolling() { stopPolling(); pollTimer = setInterval(() => cargarMensajes({ full: false }), 3500); }
  function stopPolling() { if (pollTimer) clearInterval(pollTimer); pollTimer = null; }

  // -------- ENVIAR --------
  async function enviar() {
    if (!inputEl && selectedFiles.length === 0) return;

    // 🔒 Debe estar logueado para chatear / ver susurros correctos
    const myPhone = getMyPhone();
    if (!myPhone) {
      showToast('Debes iniciar sesión para enviar mensajes.');
      // Si tienes un modal de login:
      try { document.getElementById('login-modal').style.display = 'flex'; } catch {}
      return;
    }

    const texto    = (inputEl?.value || '').trim();
    const hasFiles = selectedFiles.length > 0;

    if (!texto && !hasFiles) { inputEl?.focus(); return; }

    // 📨 Reply: solo enviamos el id REAL del backend
    const replyToId = (replyCtx && replyCtx.backendId != null) ? replyCtx.backendId : null;

    // 🤫 Whisper: debe existir destinatario SIEMPRE
    if (whisperCtx && !whisperCtx.telefono) {
      showToast('El susurro necesita un destinatario.');
      return;
    }

    if (sendBtn) sendBtn.disabled = true;

    const baseFields = {
      telefono: myPhone,
      nombre:   getMyName(),
      privado:  whisperCtx ? 1 : 0
    };
    if (texto) baseFields.texto = texto;

    // Reply
    if (replyToId != null) baseFields.reply_to = replyToId;
    if (replyCtx) {
      baseFields.reply_nombre  = replyCtx.nombre || null;
      baseFields.reply_excerpt = (replyCtx.texto || '').slice(0,150) || null;
    }

    // Whisper
    if (whisperCtx) {
      baseFields.para          = whisperCtx.telefono || '';
      baseFields.para_telefono = whisperCtx.telefono || '';
    }

    try {
      let resp, data;
      if (hasFiles) {
        const fd = new FormData();
        Object.entries(baseFields).forEach(([k,v]) => (v!=null) && fd.append(k, v));
        selectedFiles.forEach(f => fd.append('files[]', f));
        resp = await fetch(BE + 'chat.php', { method: 'POST', body: fd });
      } else {
        resp = await fetch(BE + 'chat.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json', 'Accept':'application/json' },
          body: JSON.stringify(baseFields)
        });
      }

      const raw = await resp.text();
      try { data = JSON.parse(raw); } catch { data = null; }

      // Mapea errores comunes del servidor para que se lean claro
      if (!resp.ok || !data || data.success === false) {
        let msg = (data && (data.error || data.message)) || raw || ('HTTP ' + resp.status);
        if (resp.status === 413 || /exceed/i.test(raw)) msg = 'Archivo supera el límite del servidor.';
        throw new Error(msg);
      }

      // ✅ Limpieza UI
      if (inputEl) inputEl.value = '';
      selectedFiles.length = 0; renderAttachBar();
      replyCtx = null; whisperCtx = null; renderContextBar();
      ensureNewMsgButton().style.display = 'none';
      setChatMode(null);

      // 🔄 Recargar nuevos mensajes (con bust de caché)
      await cargarMensajes({ full: false });
      list.scrollTop = list.scrollHeight;
    } catch (e) {
      console.error('[chat enviar]', e);
      showToast('❌ Error enviando el mensaje: ' + (e.message || ''));
    } finally {
      if (sendBtn) sendBtn.disabled = false;
      inputEl?.focus();
    }
  }

  // -------- ABRIR / CERRAR --------
  function openChat() {
    if (!overlay) return;
    if (!getMyPhone()) {
      showToast('Inicia sesión para usar el chat.');
      try { document.getElementById('login-modal').style.display = 'flex'; } catch {}
      return;
    }
    overlay.hidden = false;
    overlay.classList.add('open');
    overlay.setAttribute('aria-hidden', 'false');
    cargarMensajes({ full: true });  // trae ya filtrado por ?telefono=...
    startPolling();
    inputEl?.focus();
  }

  function closeChat() {
    stopPolling();
    overlay?.classList.remove('open');
    overlay?.setAttribute('aria-hidden', 'true');
    overlay && (overlay.hidden = true);
    const pop = document.querySelector('.emoji-pop'); if (pop) pop.style.display = 'none';
  }
  window.ChatTBMP = { open: openChat, close: closeChat, reload(){ cargarMensajes({ full:true }); } };

  // -------- EVENTOS UI --------
  sendBtn?.addEventListener('click', enviar);
  inputEl?.addEventListener('keydown', (e) => { if (e.key === 'Enter') { e.preventDefault(); enviar(); } });

  closeBtn?.addEventListener('click', closeChat);
  overlay?.addEventListener('click', (e) => { if (box && !box.contains(e.target)) closeChat(); });

  document.addEventListener('visibilitychange', () => {
    if (document.hidden) stopPolling();
    else if (overlay?.classList.contains('open')) startPolling();
  });

  // Hook opcional: botón flotante
  // Hook del botón chat-toggle con verificación de login
  document.getElementById('chat-toggle')?.addEventListener('click', () => {
    const isLogged = !!(
      sessionStorage.getItem('loggedPhone') ||
      (window.currentUser && window.currentUser.telefono)
    );
    if (!isLogged) {
      try { document.getElementById('login-modal').style.display = 'flex'; } catch {}
      return;
    }
    if (window.ChatTBMP && typeof window.ChatTBMP.open === 'function') {
      window.ChatTBMP.open();
    } else {
      // fallback por si aún no está inicializado
      try { openChat(); } catch {}
    }
  });

  // Atajo dev: Ctrl+/ abre el chat
  window.addEventListener('keydown', (e) => {
    if ((e.ctrlKey || e.metaKey) && e.key === '/') openChat();
  });

  // Mantener pegado al fondo cuando cargan medias
  list?.addEventListener('load', (e) => {
    const t = e.target;
    if (t && (t.tagName === 'IMG' || t.tagName === 'VIDEO')) {
      if (isAtBottom(list, 32)) list.scrollTop = list.scrollHeight;
    }
  }, true);

  // -------- “MÁS PUDIENTE” (foto/nombre desde backend) --------
  async function fetchPudiente() {
    try {
      const r = await fetch(BE + 'pudiente.php', { cache: 'no-store' });
      const j = await r.json();
      if (!j || !j.success || !j.top) return null;
      return j.top; // {id, nombre, telefono, avatarURL, title?}
    } catch { return null; }
  }
  async function refreshPudienteUI() {
    const top = await fetchPudiente();
    const img = document.getElementById('chat-president-avatar');
    const nameEl = document.getElementById('chat-president-name');
    const titleEl = document.getElementById('chat-president-title');
    const label = document.getElementById('president-photo-label');

    if (top && img) img.src = top.avatarURL || '/uploads/avatar.jpg';
    if (top && nameEl) nameEl.textContent = top.nombre || 'Presidente';
    if (titleEl) titleEl.textContent = 'El más pudiente';

    if (label) {
      const me = window.currentUser;
      const soyTop = !!(top && me && Number(me.id) === Number(top.id));
      label.style.display = soyTop ? 'inline-flex' : 'none';
    }
  }
  setInterval(refreshPudienteUI, 7000);
  refreshPudienteUI();

  // Subir nueva foto si es el top
  (function hookUpload() {
    const presInput = document.getElementById('president-photo-input');
    if (!presInput) return;
    presInput.addEventListener('change', async () => {
      if (!presInput.files || !presInput.files[0]) return;
      if (!isLogged()) { showToast('Debes iniciar sesión'); presInput.value = ''; return; }
      const fd = new FormData();
      fd.append('foto', presInput.files[0]);
      fd.append('telefono', getMyPhone());
      try {
        const r = await fetch(UPLOAD_BE  + 'upload_president.php', { method: 'POST', body: fd });
        const data = await r.json().catch(() => null);
        if (data && data.success) {
          showToast('✅ Foto actualizada');
          const img = document.getElementById('chat-president-avatar');
          if (img) { const base = img.src.split('?')[0]; img.src = base + '?v=' + Date.now(); }
        } else {
          showToast('❌ ' + (data?.error || 'No se pudo subir la foto'));
        }
      } catch { showToast('❌ Error de red subiendo foto'); }
      finally { presInput.value = ''; }
    });
  })();

})();
