<?php
// /backend/finanzas.php — devuelve resumen financiero de pedidos y canjes por rango de fechas
declare(strict_types=1);

// Aseguramos la zona horaria a Caracas para que los cortes de día sean consistentes
date_default_timezone_set('America/Caracas');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
header('Access-Control-Allow-Origin: *');

// Protecciones de sesión y CSRF
require_once __DIR__ . '/guard.php';
require_admin();
// Only require CSRF for export (download), not for JSON GET
// If export is requested, ensure token for write-like action
if (isset($_GET['export'])) {
    require_csrf_for_write();
}

require_once __DIR__ . '/db.php';
$mysqli->set_charset('utf8mb4');
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

/**
 * Devuelve la salida JSON y finaliza.
 */
function fn_out(array $arr, int $code = 200): void {
    http_response_code($code);
    echo json_encode($arr, JSON_UNESCAPED_UNICODE);
    exit;
}

// Obtener fechas de la query; por defecto, hoy
$desde = trim($_GET['desde'] ?? '');
$hasta = trim($_GET['hasta'] ?? '');

if ($desde === '') {
    $desde = date('Y-m-d');
}
if ($hasta === '') {
    $hasta = $desde;
}

// Normalizar a rangos de 00:00:00 a 23:59:59
$desdeDt = $desde . ' 00:00:00';
$hastaDt = $hasta . ' 23:59:59';

try {
    $summary = [
        'pedidos'     => ['total' => 0.0, 'count' => 0],
        'solicitudes' => ['total' => 0.0, 'count' => 0]
    ];
    $records = [];

    // ---- Pedidos: Sumatoria de USD y conteo por día ----
    $stmt = $mysqli->prepare("SELECT DATE(processedAt) AS d, SUM(montoUsd) AS total, COUNT(*) AS c FROM pedidos WHERE LOWER(COALESCE(estado,'pendiente'))='aprobado' AND processedAt BETWEEN ? AND ? GROUP BY DATE(processedAt)");
    $stmt->bind_param('ss', $desdeDt, $hastaDt);
    $stmt->execute();
    $res = $stmt->get_result();
    $pedidosByDate = [];
    while ($row = $res->fetch_assoc()) {
        $d = $row['d'] ?? '';
        $total = (float)($row['total'] ?? 0);
        $cnt = (int)($row['c'] ?? 0);
        $pedidosByDate[$d] = ['total' => $total, 'count' => $cnt];
        $summary['pedidos']['total'] += $total;
        $summary['pedidos']['count'] += $cnt;
    }
    $stmt->close();

    // ---- Solicitudes (canjes): Sumatoria de coins y conteo por día ----
    $stmt = $mysqli->prepare("SELECT DATE(approvedAt) AS d, SUM(monto) AS total, COUNT(*) AS c FROM solicitudes WHERE LOWER(COALESCE(estado,'pendiente')) IN ('aprobada','aprobado') AND approvedAt BETWEEN ? AND ? GROUP BY DATE(approvedAt)");
    $stmt->bind_param('ss', $desdeDt, $hastaDt);
    $stmt->execute();
    $res = $stmt->get_result();
    $solByDate = [];
    while ($row = $res->fetch_assoc()) {
        $d = $row['d'] ?? '';
        $total = (float)($row['total'] ?? 0);
        $cnt = (int)($row['c'] ?? 0);
        $solByDate[$d] = ['total' => $total, 'count' => $cnt];
        $summary['solicitudes']['total'] += $total;
        $summary['solicitudes']['count'] += $cnt;
    }
    $stmt->close();

    // Generar registros por día dentro del rango
    $start = new DateTime($desde);
    $end   = new DateTime($hasta);
    while ($start <= $end) {
        $dStr = $start->format('Y-m-d');
        $ped = $pedidosByDate[$dStr] ?? ['total' => 0.0, 'count' => 0];
        $sol = $solByDate[$dStr] ?? ['total' => 0.0, 'count' => 0];
        $records[] = [
            'date'        => $dStr,
            'pedidos'     => $ped,
            'solicitudes' => $sol
        ];
        $start->modify('+1 day');
    }

    // Ordenar los registros por fecha descendente para mostrar primero los
    // movimientos más recientes. El formato Y-m-d permite ordenar como cadena.
    usort($records, function($a, $b) {
        return strcmp($b['date'], $a['date']);
    });

    // Si se solicita exportar a CSV o Excel, generar archivo
    $export = strtolower(trim($_GET['export'] ?? ''));
    if ($export === 'csv' || $export === 'excel') {
        // Generar título de archivo
        $ext = $export === 'csv' ? 'csv' : 'xls';
        $filename = 'finanzas_' . $desde . '_al_' . $hasta . '.' . $ext;
        // Preparar header
        if ($export === 'csv') {
            header('Content-Type: text/csv; charset=utf-8');
        } else {
            // Para Excel usamos tipo de contenido compatible con Excel 97-2003
            header('Content-Type: application/vnd.ms-excel; charset=utf-8');
        }
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        // Imprimir encabezados
        $output = fopen('php://output', 'w');
        // Fila encabezado
        fputcsv($output, ['Fecha', 'Recargas USD', '# Recargas', 'Canjes Coins', '# Canjes']);
        foreach ($records as $rec) {
            $d   = $rec['date'];
            $ped = $rec['pedidos'];
            $sol = $rec['solicitudes'];
            fputcsv($output, [
                $d,
                $ped['total'] ?? 0,
                $ped['count'] ?? 0,
                $sol['total'] ?? 0,
                $sol['count'] ?? 0,
            ]);
        }
        fclose($output);
        exit;
    }
    fn_out([
        'success' => true,
        'summary' => $summary,
        'records' => $records
    ]);
} catch (Throwable $e) {
    fn_out(['success' => false, 'error' => 'Error: ' . $e->getMessage()], 500);
}