<?php
// /backend/admin.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
header('Access-Control-Allow-Origin: *');

// Protecciones de acceso
require_once __DIR__ . '/guard.php';
require_admin();
require_csrf_for_write();

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once __DIR__ . '/db.php';   // Debe definir $mysqli = new mysqli(...)
$mysqli->set_charset('utf8mb4');

const T_USUARIOS    = 'usuarios';
const T_PEDIDOS     = 'pedidos';
const T_SOLICITUDES = 'solicitudes';
const COIN_RATE_USD = 3.3;

// ---------- util salida JSON ----------
function jout($data, int $code = 200) {
  while (ob_get_level()) { ob_end_clean(); }
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}
function ok($msg, $extra = [])  { jout(array_merge(['success'=>true,  'message'=>$msg], $extra)); }
function fail($msg, $code=400, $extra=[]) { jout(array_merge(['success'=>false, 'error'=>$msg], $extra), $code); }

function body_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return $_POST ?? [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : ($_POST ?? []);
}
function tableExists(mysqli $db, string $name): bool {
  $name = $db->real_escape_string($name);
  $res = $db->query("SHOW TABLES LIKE '$name'");
  return $res && $res->num_rows > 0;
}

/* =======================
 * GET ?tipo=...
 * ======================= */
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
  $tipo = strtolower($_GET['tipo'] ?? '');

  try {
    switch ($tipo) {
      case 'ping':
        ok('pong', ['time' => date('c')]);
        break;

      case 'usuarios': {
        if (!tableExists($mysqli, T_USUARIOS)) jout([]);
        $sql = "SELECT id, nombre, telefono, cumanaCoins, maxCumanaCoins,
                       IFNULL(banned,0) AS banned, fechaRegistro, avatarURL
                FROM ".T_USUARIOS."
                ORDER BY COALESCE(fechaRegistro,'1970-01-01') DESC";
        $rows = $mysqli->query($sql)->fetch_all(MYSQLI_ASSOC);
        jout($rows);
      }
      break;

      case 'pedidos': {
        if (!tableExists($mysqli, T_PEDIDOS) || !tableExists($mysqli, T_USUARIOS)) jout([]);

        // Filtros opcionales: ?estado=pendiente|aprobado|rechazado | ?include_rechazados=1
        $estado = strtolower(trim($_GET['estado'] ?? ''));
        $inclR  = (int)($_GET['include_rechazados'] ?? 0);

        if ($estado !== '') {
          $where = "WHERE LOWER(COALESCE(p.estado,'pendiente')) = '" . $mysqli->real_escape_string($estado) . "'";
        } else {
          $where = ($inclR === 1)
            ? "WHERE LOWER(COALESCE(p.estado,'pendiente')) IN ('pendiente','aprobado','rechazado')"
            : "WHERE LOWER(COALESCE(p.estado,'pendiente')) IN ('pendiente','aprobado')";
        }

        $sql = "SELECT
                  p.id, p.userId,
                  u.nombre, u.telefono,
                  p.montoUsd, COALESCE(p.referencia (Ref),1) AS referencia (Ref),
                  LOWER(COALESCE(p.estado,'pendiente')) AS estado,
                  p.order_code,
                  p.created_at AS createdAt,
                  p.fecha      AS fecha,
                  p.processedAt, p.processedBy
                FROM ".T_PEDIDOS." p
                JOIN ".T_USUARIOS." u ON u.id = p.userId
                $where
                ORDER BY (LOWER(COALESCE(p.estado,'pendiente'))='pendiente') DESC,
                         COALESCE(p.processedAt, p.created_at) DESC,
                         p.id DESC
                LIMIT 500";
        $rows = $mysqli->query($sql)->fetch_all(MYSQLI_ASSOC);
        jout($rows);
      }
      break;

      case 'solicitudes': {
        if (!tableExists($mysqli, T_SOLICITUDES) || !tableExists($mysqli, T_USUARIOS)) jout([]);
        $sql = "SELECT
                  s.id, s.userId,
                  u.nombre, u.telefono,
                  s.tipo, s.monto,
                  COALESCE(s.estado,'pendiente') AS estado,
                  s.canje_code,
                  s.createdAt
                FROM ".T_SOLICITUDES." s
                LEFT JOIN ".T_USUARIOS." u ON u.id = s.userId
                ORDER BY (LOWER(COALESCE(s.estado,'pendiente'))='pendiente') DESC,
                         s.createdAt DESC,
                         s.id DESC";
        $rows = $mysqli->query($sql)->fetch_all(MYSQLI_ASSOC);
        jout($rows);
      }
      break;

      default:
        jout([]); // siempre JSON
    }
  } catch (Throwable $e) {
    fail('SQL: ' . $e->getMessage(), 500);
  }
}

/* =======================
 * POST acciones (JSON)
 * ======================= */
$post   = body_json();
$accion = $post['accion'] ?? null;

// --- Aprobar pedido ---
if ($accion === 'aprobar_pedido') {
  if (!tableExists($mysqli, T_PEDIDOS) || !tableExists($mysqli, T_USUARIOS)) fail('Tablas requeridas no existen',500);
  $id = (int)($post['id'] ?? 0);
  if ($id<=0) fail('ID inválido');

  $mysqli->begin_transaction();
  try {
      // Si el admin envía un montoUsd override (>0), úsalo


    $st = $mysqli->prepare("SELECT id, userId, montoUsd, COALESCE(estado,'pendiente') AS estado FROM ".T_PEDIDOS." WHERE id=? FOR UPDATE");
    $st->bind_param('i',$id); $st->execute(); $p = $st->get_result()->fetch_assoc(); $st->close();
    if (!$p) { $mysqli->rollback(); fail('Pedido no encontrado',404); }
    if (strtolower($p['estado'])!=='pendiente'){ $mysqli->commit(); ok('Pedido ya procesado',['estado'=>$p['estado']]); }
    // Si el admin envía un montoUsd override (>0), úsalo
$overrideUsd = isset($post['montoUsd']) ? (float)$post['montoUsd'] : 0.0;
if ($overrideUsd > 0) {
  $p['montoUsd'] = $overrideUsd;
}


    $st = $mysqli->prepare("SELECT cumanaCoins, maxCumanaCoins FROM ".T_USUARIOS." WHERE id=? FOR UPDATE");
    $st->bind_param('i',$p['userId']); $st->execute(); $u = $st->get_result()->fetch_assoc(); $st->close();
    if (!$u){ $mysqli->rollback(); fail('Usuario no encontrado',404); }

    $coinsActual = (int)$u['cumanaCoins'];
    $maxActual   = (int)$u['maxCumanaCoins'];
    $acreditar   = (int)round(((float)$p['montoUsd']) * COIN_RATE_USD);
    if ($acreditar<=0){ $mysqli->rollback(); fail('Monto inválido'); }

    $nuevo   = $coinsActual + $acreditar;
    $nuevoMx = max($maxActual, $nuevo);

    $st = $mysqli->prepare("UPDATE ".T_USUARIOS." SET cumanaCoins=?, maxCumanaCoins=? WHERE id=?");
    $st->bind_param('iii',$nuevo,$nuevoMx,$p['userId']); $st->execute(); $st->close();

    $by = trim((string)($post['by'] ?? 'admin'));
   $st = $mysqli->prepare("UPDATE ".T_PEDIDOS." SET montoUsd=?, estado='aprobado', processedAt=NOW(), processedBy=? WHERE id=?");
$st->bind_param('dsi', $p['montoUsd'], $by, $id);

    $st->execute();
    $st->close();

    $mysqli->commit();
    ok('Pedido aprobado', ['userId'=>$p['userId'],'coinsSumados'=>$acreditar,'saldoNuevo'=>$nuevo]);
  } catch (Throwable $e) {
    $mysqli->rollback(); fail('Error al aprobar pedido: '.$e->getMessage(),500);
  }
}

// --- Rechazar pedido ---
if ($accion === 'rechazar_pedido') {
  if (!tableExists($mysqli, T_PEDIDOS)) fail('Tabla pedidos no existe', 500);
  $id = (int)($post['id'] ?? 0);
  if ($id <= 0) fail('ID inválido');

  $by = trim((string)($post['by'] ?? 'admin'));

  $mysqli->begin_transaction();
  try {
    $st = $mysqli->prepare("SELECT COALESCE(estado,'pendiente') AS estado FROM ".T_PEDIDOS." WHERE id=? FOR UPDATE");
    $st->bind_param('i',$id);
    $st->execute();
    $p = $st->get_result()->fetch_assoc();
    $st->close();

    if (!$p) { $mysqli->rollback(); fail('Pedido no encontrado', 404); }

    $est = strtolower($p['estado']);
    if ($est !== 'pendiente') {
      $mysqli->commit();
      ok('Pedido ya procesado', ['id'=>$id, 'estado'=>$p['estado']]);
    }

    $st = $mysqli->prepare("UPDATE ".T_PEDIDOS." SET estado='rechazado', processedAt=NOW(), processedBy=? WHERE id=?");
    $st->bind_param('si', $by, $id);
    $st->execute();
    $st->close();

    $mysqli->commit();
    ok('Pedido rechazado', ['id'=>$id, 'estado'=>'rechazado']);
  } catch (Throwable $e) {
    $mysqli->rollback();
    fail('Error al rechazar pedido: '.$e->getMessage(), 500);
  }
}

// --- Aprobar canje ---
if ($accion === 'aprobar_canje') {
  if (!tableExists($mysqli, T_SOLICITUDES) || !tableExists($mysqli, T_USUARIOS)) fail('Tablas requeridas no existen',500);
  $id = (int)($post['id'] ?? 0);
  if ($id<=0) fail('ID inválido');

  $mysqli->begin_transaction();
  try {
    $st = $mysqli->prepare("
      SELECT s.id, s.userId, s.tipo, s.monto, COALESCE(s.estado,'pendiente') AS estado,
             u.cumanaCoins, u.maxCumanaCoins
      FROM ".T_SOLICITUDES." s
      JOIN ".T_USUARIOS." u ON u.id = s.userId
      WHERE s.id=? FOR UPDATE
    ");
    $st->bind_param('i',$id); $st->execute(); $row = $st->get_result()->fetch_assoc(); $st->close();

    if (!$row){ $mysqli->rollback(); fail('Solicitud no encontrada',404); }
    if (strtolower($row['estado'])!=='pendiente'){ $mysqli->commit(); ok('Solicitud ya procesada',['estado'=>$row['estado']]); }
    if ($row['tipo'] !== 'canjear') { $mysqli->rollback(); fail('Tipo inválido'); }

    $monto = (int)$row['monto'];
    $coins = (int)$row['cumanaCoins'];
    if ($monto<=0){ $mysqli->rollback(); fail('Monto inválido'); }
    if ($coins < $monto){ $mysqli->rollback(); fail('Saldo insuficiente'); }

    $nuevo = $coins - $monto;

    $st = $mysqli->prepare("UPDATE ".T_USUARIOS." SET cumanaCoins=? WHERE id=?");
    $st->bind_param('ii',$nuevo,$row['userId']); $st->execute(); $st->close();

    $st = $mysqli->prepare("UPDATE ".T_SOLICITUDES." SET estado='aprobada', approvedAt=NOW(), approvedBy='admin' WHERE id=?");
    $st->bind_param('i',$id); $st->execute(); $st->close();

    $mysqli->commit();
    ok('Solicitud aprobada', ['id'=>$id,'saldoNuevo'=>$nuevo]);
  } catch (Throwable $e) {
    $mysqli->rollback(); fail('Error al aprobar canje: '.$e->getMessage(),500);
  }
}

// --- Rechazar canje ---
if ($accion === 'rechazar_canje') {
  if (!tableExists($mysqli, T_SOLICITUDES)) fail('Tabla solicitudes no existe',500);
  $id = (int)($post['id'] ?? 0);
  if ($id<=0) fail('ID inválido');

  $mysqli->begin_transaction();
  try {
    $st = $mysqli->prepare("SELECT COALESCE(estado,'pendiente') AS estado FROM ".T_SOLICITUDES." WHERE id=? FOR UPDATE");
    $st->bind_param('i',$id); $st->execute(); $s = $st->get_result()->fetch_assoc(); $st->close();

    if (!$s) { $mysqli->rollback(); fail('Solicitud no encontrada',404); }
    if (strtolower($s['estado']) !== 'pendiente') {
      $mysqli->rollback(); fail('La solicitud no está pendiente');
    }

    $st = $mysqli->prepare("UPDATE ".T_SOLICITUDES."
                            SET estado='rechazada', approvedAt=NOW(), approvedBy='admin'
                            WHERE id=?");
    $st->bind_param('i',$id); $st->execute(); $st->close();

    $mysqli->commit();
    ok('Solicitud rechazada', ['id'=>$id, 'estado'=>'rechazada']);
  } catch (Throwable $e) {
    $mysqli->rollback(); fail('Error al rechazar canje: '.$e->getMessage(),500);
  }
}

// --- si nada coincidió ---
fail('Operación no soportada');
