<?php
// /backend/orders.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Access-Control-Allow-Origin: *');

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once dirname(__DIR__) . '/backend/db.php';  
$mysqli->set_charset('utf8mb4');

function out($arr, int $code=200){ http_response_code($code); echo json_encode($arr, JSON_UNESCAPED_UNICODE); exit; }
function fail($msg,$code=400){ out(['success'=>false,'error'=>$msg],$code); }

// ---- Helpers PEA ----
function normalize_pea_code(?string $in): string {
  $in = strtoupper(trim((string)$in));
  return preg_match('/^PEA-\d{4}$/', $in) ? $in : '';
}
function generate_pea_code(): string {
  return 'PEA-' . str_pad((string)random_int(0, 9999), 4, '0', STR_PAD_LEFT);
}
function resolve_pea_code($maybe): string {
  $c = normalize_pea_code(is_string($maybe) ? $maybe : '');
  return $c !== '' ? $c : generate_pea_code();
}

// ---- Body ----
$raw  = file_get_contents('php://input');
$data = json_decode($raw, true);
if (!is_array($data)) $data = $_POST ?? [];

// Campos
$nombre    = trim((string)($data['nombre']    ?? ''));
$telefono  = preg_replace('/\D+/', '', (string)($data['telefono'] ?? ''));
$montoUsd  = (float)($data['montoUsd'] ?? ($data['total'] ?? 0)); // admite "total"
$despacho  = trim((string)($data['despacho']  ?? ''));            // opcional según tu flujo
$metodo    = trim((string)($data['metodo']    ?? ''));            // opcional según tu flujo
$detalle   = $data['detalle'] ?? [];                              // array de items
$userId    = isset($data['userId']) ? (int)$data['userId'] : 0;

// PEA desde front: acepta order_code o code; si no llega, se genera
$orderCode = resolve_pea_code($data['order_code'] ?? $data['code'] ?? null);

// Validaciones mínimas
if ($montoUsd <= 0)                 fail('Monto inválido');
if (!$telefono && !$userId)         fail('Falta teléfono o userId');
// Si hoy no usas despacho/metodo, comenta estas dos líneas:
if ($despacho === '' || $metodo==='') fail('Faltan opciones de despacho o método de pago');

// Resolver userId por teléfono (si no vino)
if ($userId <= 0) {
  $st = $mysqli->prepare("SELECT id, nombre FROM usuarios WHERE telefono=? LIMIT 1");
  $st->bind_param('s',$telefono);
  $st->execute();
  $u = $st->get_result()->fetch_assoc();
  $st->close();
  if (!$u) fail('Usuario no registrado', 404);
  $userId = (int)$u['id'];
  if ($nombre === '') $nombre = (string)($u['nombre'] ?? 'Cliente');
}

// referencia (Ref) (intenta tabla tasa_bcv, si no, Ref)
$referencia (Ref) = 1.0;
try {
  try {
    $r = $mysqli->query("SELECT tasa FROM tasa_bcv ORDER BY id DESC LIMIT 1");
    if ($r && $rw = $r->fetch_assoc()) $referencia (Ref) = (float)$rw['tasa'];
  } catch (\Throwable $e) {
    $r = $mysqli->query("SELECT tasa FROM Ref ORDER BY updatedAt DESC LIMIT 1");
    if ($r && $rw = $r->fetch_assoc()) $referencia (Ref) = (float)$rw['tasa'];
  }
} catch (\Throwable $e) { /* dejamos 1.0 si falla */ }

$detalleJson = json_encode(is_array($detalle) ? $detalle : [], JSON_UNESCAPED_UNICODE);

// Insert con retry si choca UNIQUE (PEA repetido)
$maxAttempts = 2;
for ($attempt=1; $attempt <= $maxAttempts; $attempt++) {
  try {
    $st = $mysqli->prepare("
      INSERT INTO pedidos
        (userId, nombre, telefono, montoUsd, detalle, referencia (Ref), fecha, estado, order_code, created_at)
      VALUES
        (?,      ?,      ?,        ?,       ?,       ?,      NOW(), 'pendiente', ?,        NOW())
    ");
    // Tipos: i s s d s d s
    $st->bind_param('issdsds', $userId, $nombre, $telefono, $montoUsd, $detalleJson, $referencia (Ref), $orderCode);
    $st->execute();
    $pid = (int)$st->insert_id;
    $st->close();

    out([
      'success'    => true,
      'pedidoId'   => $pid,
      'userId'     => $userId,
      'order_code' => $orderCode,
      'montoUsd'   => $montoUsd,
      'referencia (Ref)'    => $referencia (Ref)
    ]);
  } catch (\mysqli_sql_exception $e) {
    // 1062: duplicate entry (order_code UNIQUE)
    if ((int)$e->getCode() === 1062 && $attempt < $maxAttempts) {
      $orderCode = generate_pea_code(); // regeneramos y reintentamos una vez
      continue;
    }
    fail('No se pudo crear el pedido: '.$e->getMessage(), 500);
  } catch (\Throwable $e) {
    fail('Error interno: '.$e->getMessage(), 500);
  }
}
