<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

/* =================== DEBUG / ERRORES =================== */
$DEBUG = isset($_GET['debug']) || isset($_POST['debug']);
error_reporting(E_ALL);
ini_set('display_errors', $DEBUG ? '1' : '0');

/* Devuelve JSON si ocurre un fatal y no se imprimió nada */
register_shutdown_function(function () use ($DEBUG) {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    $msg = $DEBUG ? ($e['message'].' @'.$e['file'].':'.$e['line']) : 'Fatal error';
    echo json_encode(['success'=>false,'error'=>$msg]);
  }
});

/* =================== CREDENCIALES DB =================== */
if (!defined('DB_HOST')) define('DB_HOST', 'localhost');        // prueba '127.0.0.1' si falla
if (!defined('DB_NAME')) define('DB_NAME', 'tanbella_database');
if (!defined('DB_USER')) define('DB_USER', 'tanbella_user');
if (!defined('DB_PASS')) define('DB_PASS', 'Adsolutions21*');

$pdo  = null;  // preferido
$conn = null;  // fallback mysqli

/* Si tienes db.php y no rompe, se cargará primero */
@require_once __DIR__ . '/../backend/db.php';
if (!isset($pdo))  $pdo  = null;
if (!isset($conn)) $conn = null;

/* ----- Conexión PDO (si existe pdo_mysql) ----- */
if (!$pdo && !$conn) {
  try {
    $pdo = new PDO(
      'mysql:host='.DB_HOST.';dbname='.DB_NAME.';charset=utf8mb4',
      DB_USER,
      DB_PASS,
      [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
      ]
    );
    $pdo->exec("SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci");
  } catch (Throwable $e) {
    $pdo = null; /* seguimos con mysqli si está disponible */
  }
}

/* ----- Conexión mysqli (solo si la extensión existe) ----- */
if (!$pdo && !$conn && extension_loaded('mysqli')) {
  $tmp = @new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
  if ($tmp && !$tmp->connect_errno) {
    $tmp->set_charset('utf8mb4');
    $conn = $tmp;
  }
}

/* =================== HELPERS DB =================== */
function db_is_pdo(): bool    { return ($GLOBALS['pdo']  instanceof PDO); }
function db_is_mysqli(): bool { return ($GLOBALS['conn'] instanceof mysqli); }

function db_query(string $sql, array $params = []) {
  if (db_is_pdo()) {
    $st = $GLOBALS['pdo']->prepare($sql);
    $st->execute($params);
    return $st;
  }
  if (db_is_mysqli()) {
    $mysqli = $GLOBALS['conn'];
    $st = $mysqli->prepare($sql);
    if (!$st) throw new RuntimeException('mysqli prepare: '.$mysqli->error);

    if ($params) {
      $types = '';
      $vals  = [];
      foreach ($params as $v) {
        if (is_int($v))        $types .= 'i';
        elseif (is_float($v))  $types .= 'd';
        else                   $types .= 's';
        $vals[] = $v;
      }
      $bind = [];
      $bind[] = &$types;
      foreach ($vals as $k => $v) $bind[] = &$vals[$k];
      if (!call_user_func_array([$st, 'bind_param'], $bind)) {
        throw new RuntimeException('mysqli bind_param falló');
      }
    }
    if (!$st->execute()) throw new RuntimeException('mysqli execute: '.$st->error);
    return $st;
  }
  throw new RuntimeException('No DB connection available');
}
function db_last_id(): int {
  if (db_is_pdo())    return (int)$GLOBALS['pdo']->lastInsertId();
  if (db_is_mysqli()) return (int)$GLOBALS['conn']->insert_id;
  return 0;
}
function db_fetch_all($st): array {
  if (db_is_pdo())    return $st->fetchAll(PDO::FETCH_ASSOC);
  if (db_is_mysqli()) { $res = $st->get_result(); return $res ? $res->fetch_all(MYSQLI_ASSOC) : []; }
  return [];
}

/* =================== UTILIDADES =================== */
function b($v){ return $v==='1'||$v===1||$v===true||strtolower((string)$v)==='true'; }
function jerr(string $msg, int $code=400){ http_response_code($code); echo json_encode(['success'=>false,'error'=>$msg]); exit; }
function short_txt($s, int $n=200){ return function_exists('mb_substr') ? mb_substr((string)$s,0,$n) : substr((string)$s,0,$n); }

/* =================== PING =================== */
if (isset($_GET['ping'])) {
  echo json_encode([
    'ok'     => (db_is_pdo() || db_is_mysqli()),
    'driver' => db_is_pdo() ? 'pdo' : (db_is_mysqli() ? 'mysqli' : 'none'),
    'host'   => DB_HOST,
    'db'     => DB_NAME,
    'time'   => date('c'),
  ]);
  exit;
}

/* =================== RUTEO =================== */
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

/* ------------------ POST (enviar mensaje) ------------------ */
if ($method === 'POST') {
  $ct = $_SERVER['CONTENT_TYPE'] ?? '';
  $isMultipart = stripos($ct, 'multipart/form-data') !== false;

  if ($isMultipart) {
    $telefono = trim($_POST['telefono'] ?? '');
    $nombre   = trim($_POST['nombre']   ?? '');
    $texto    = trim($_POST['texto']    ?? '');
    $reply_to = isset($_POST['reply_to']) ? (int)$_POST['reply_to'] : null;
    $privado  = b($_POST['privado'] ?? 0) ? 1 : 0;
    $para     = trim($_POST['para'] ?? ($_POST['para_telefono'] ?? ''));
  } else {
    $raw  = file_get_contents('php://input') ?: '{}';
    $body = json_decode($raw, true) ?: [];
    $telefono = trim($body['telefono'] ?? '');
    $nombre   = trim($body['nombre']   ?? '');
    $texto    = trim($body['texto']    ?? '');
    $reply_to = isset($body['reply_to']) ? (int)$body['reply_to'] : null;
    $privado  = b($body['privado'] ?? 0) ? 1 : 0;
    $para     = trim($body['para'] ?? ($body['para_telefono'] ?? ''));
  }

  if ($telefono === '')          jerr('Falta telefono');
  if ($privado && $para === '')  jerr('Falta destinatario para susurro');
  if ($nombre === '')            $nombre = 'Anon';

  if (!db_is_pdo() && !db_is_mysqli()) jerr('Sin conexión a la base de datos', 500);

  /* Insert */
  try {
    db_query(
      "INSERT INTO chat (telefono,nombre,texto,reply_to,privado,para_telefono,fecha)
       VALUES (?,?,?,?,?,?,NOW())",
      [$telefono,$nombre,$texto,$reply_to,$privado,($para ?: null)]
    );
    $mid = db_last_id();
  } catch (Throwable $e) {
    jerr('Insert chat: '.$e->getMessage(), 500);
  }

  /* Adjuntos (opcional) -> /uploads/chat */
  $filesOut = [];
  if (!empty($_FILES['files']) && is_array($_FILES['files']['name'])) {
    $root = __DIR__ . '/../uploads';
    if (!is_dir($root)) @mkdir($root, 0775, true);
    $chatDir = $root . '/chat';
    if (!is_dir($chatDir)) @mkdir($chatDir, 0775, true);

    for ($i=0; $i<count($_FILES['files']['name']); $i++) {
      if (($_FILES['files']['error'][$i] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) continue;
      $tmp  = $_FILES['files']['tmp_name'][$i];
      $orig = basename((string)$_FILES['files']['name'][$i]);
      $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
      $base = preg_replace('/[^a-z0-9\-_]+/i', '_', pathinfo($orig, PATHINFO_FILENAME));
      $fname = $mid.'_'.time().'_'.$i.($ext?'.'.$ext:'');
      $dest  = $chatDir.'/'.$fname;

      if (@move_uploaded_file($tmp, $dest)) {
        $mime = function_exists('mime_content_type') ? @mime_content_type($dest) : 'application/octet-stream';
        $rel  = '/uploads/chat/'.$fname;
        $size = (int)@filesize($dest);

        try {
          db_query(
            "INSERT INTO chat_adjuntos (chat_id,file_name,file_path,mime_type,file_size,created_at)
             VALUES (?,?,?,?,?,NOW())",
            [$mid,$orig,$rel,$mime,$size]
          );
        } catch (Throwable $e) { /* no aborta */ }
        $filesOut[] = ['name'=>$orig,'path'=>$rel,'mime'=>$mime,'size'=>$size];
      }
    }
  }

  /* Preview reply (opcional) */
  $reply = null;
  if ($reply_to) {
    try {
      $rs = db_query("SELECT id,nombre,texto FROM chat WHERE id=?", [$reply_to]);
      $r  = db_fetch_all($rs);
      if ($r) $reply = ['id'=>(int)$r[0]['id'],'nombre'=>$r[0]['nombre'],'texto'=>short_txt($r[0]['texto'],200)];
    } catch (Throwable $e) {}
  }

  echo json_encode([
    'success'=>true,
    'message'=>[
      'id'=>$mid,'telefono'=>$telefono,'nombre'=>$nombre,'texto'=>$texto,'fecha'=>date('c'),
      'reply_to'=>$reply_to,'reply'=>$reply,
      'privado'=>$privado,'para'=>$para,
      'adjuntos'=>$filesOut
    ]
  ]);
  exit;
}

/* ------------------ GET (listar mensajes) ------------------ */
if ($method === 'GET') {
  $tel     = trim($_GET['telefono'] ?? '');
  $admin   = b($_GET['admin'] ?? 0);
  $limit   = (int)($_GET['limit'] ?? 120);
  $sinceId = isset($_GET['since_id']) ? (int)$_GET['since_id'] : null;
  if ($limit < 1 || $limit > 500) $limit = 120;

  if (!db_is_pdo() && !db_is_mysqli()) jerr('Sin conexión a la base de datos', 500);

  try {
    if ($admin) {
      $sql = "SELECT c.*, r.nombre AS reply_nombre, r.texto AS reply_texto
              FROM chat c
              LEFT JOIN chat r ON r.id = c.reply_to
              ".($sinceId ? "WHERE c.id > ? " : "")."
              ORDER BY c.id ASC
              LIMIT ?";
      $p = [];
      if ($sinceId) $p[] = $sinceId;
      $p[] = $limit;
      $st = db_query($sql, $p);
    } else {
      $sql = "SELECT c.*, r.nombre AS reply_nombre, r.texto AS reply_texto
              FROM chat c
              LEFT JOIN chat r ON r.id = c.reply_to
              WHERE (c.privado=0 ".($sinceId ? "AND c.id > ?" : "").")
                 OR  (c.privado=1 AND (c.telefono=? OR c.para_telefono=?) ".($sinceId ? "AND c.id > ?" : "").")
              ORDER BY c.id ASC
              LIMIT ?";
      $p = [];
      if ($sinceId) { $p[] = $sinceId; }
      $p[] = $tel; $p[] = $tel;
      if ($sinceId) $p[] = $sinceId;
      $p[] = $limit;
      $st = db_query($sql, $p);
    }
    $rows = db_fetch_all($st);
  } catch (Throwable $e) {
    jerr('Select chat: '.$e->getMessage(), 500);
  }

  /* Adjuntos por lote */
  $ids = array_map(fn($r)=>(int)$r['id'], $rows);
  $atts = [];
  if ($ids) {
    $place = implode(',', array_fill(0, count($ids), '?'));
    try {
      $sa = db_query("SELECT * FROM chat_adjuntos WHERE chat_id IN ($place)", $ids);
      $allA = db_fetch_all($sa);
      foreach ($allA as $a) {
        $atts[(int)$a['chat_id']][] = [
          'name'=>$a['file_name'],
          'path'=>$a['file_path'],
          'mime'=>$a['mime_type'],
          'size'=>(int)$a['file_size'],
        ];
      }
    } catch (Throwable $e) { /* silencioso */ }
  }

  $out = [];
  foreach ($rows as $r) {
    $out[] = [
      'id'       => (int)$r['id'],
      'telefono' => $r['telefono'],
      'nombre'   => $r['nombre'],
      'texto'    => $r['texto'],
      'fecha'    => $r['fecha'] ?? null,
      'reply_to' => $r['reply_to'] ? (int)$r['reply_to'] : null,
      'reply'    => $r['reply_to'] ? [
        'id'    => (int)$r['reply_to'],
        'nombre'=> $r['reply_nombre'],
        'texto' => short_txt($r['reply_texto'] ?? '', 200)
      ] : null,
      'privado'  => (int)$r['privado'],
      'para'     => $r['para_telefono'],
      'adjuntos' => $atts[(int)$r['id']] ?? [],
    ];
  }
  echo json_encode($out);
  exit;
}

/* ------------------ 405 ------------------ */
http_response_code(405);
echo json_encode(['success'=>false,'error'=>'Método no permitido']);
