(function () {
  window.BACKEND = window.BACKEND || '/api/';
  const BE = (window.BACKEND || '/api/').replace(/\/?$/, '/');

  const IMAGE_UPLOAD_BASE = '/uploads/';
  const PLACEHOLDER_IMG = '/imgproductos/placeholder.png';

  // Estado global
  window.data = {};
  window.categoryLabels = {};
  window.cart = {};
  window.tasaBCV = 1;
  window.currentUser = null;

 
  
  const DEFAULT_CATEGORY_LABELS = {
    'base-whisky': 'Base Whisky',
    'almuerzo': 'Almuerzo',
    'comida-rapida': 'Comida Rapida',
    'vodka-anis': 'Vodka Anis',
    'cerveza': 'Cerveza',
    'jugos': 'Jugos',
    'whisky': 'Whisky',
    'ron': 'Ron',
    'tequila': 'Tequila',
    'sangria-vinos': 'Sangria Vinos'
  };

  const WA_TARGET = 'https://wa.me/584122775500';

  // === UTILIDADES ===
  function normalizeForKey(s) {
    if (!s && s !== 0) return '';
    return String(s)
      .normalize('NFD').replace(/[\u0300-\u036f]/g, '')
      .toLowerCase().replace(/\s+/g, ' ').trim()
      .replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
  }
  function joinUrl(base, path) {
    if (!base) return path || '';
    if (!path) return base;
    return base.replace(/\/+$/, '') + '/' + path.replace(/^\/+/, '');
  }
  function showNeonAlert(msg) {
    const el = document.getElementById('neon-alert');
    if (!el) { alert(msg); return; }
    el.textContent = msg;
    el.style.display = 'flex';
    el.classList.remove('show');
    void el.offsetWidth;
    el.classList.add('show');
    el.addEventListener('animationend', function handler() {
      el.classList.remove('show');
      el.style.display = 'none';
      el.removeEventListener('animationend', handler);
    });
  }
  window.showNeonAlert = showNeonAlert;

  // Overlay “Procesando…”
  function mostrarCargando(texto = '⏳ Procesando su solicitud...') {
    let overlay = document.getElementById('overlayCargando');
    if (!overlay) {
      overlay = document.createElement('div');
      overlay.id = 'overlayCargando';
      overlay.style.position = 'fixed';
      overlay.style.inset = '0';
      overlay.style.background = 'rgba(0,0,0,0.6)';
      overlay.style.display = 'flex';
      overlay.style.alignItems = 'center';
      overlay.style.justifyContent = 'center';
      overlay.style.zIndex = '9999';
      overlay.style.color = '#fff';
      overlay.style.fontSize = '1.2rem';
      document.body.appendChild(overlay);
    }
    overlay.textContent = texto;
    overlay.style.display = 'flex';
  }
  function ocultarCargando() {
    const overlay = document.getElementById('overlayCargando');
    if (overlay) overlay.style.display = 'none';
  }

  // Imagen de productos
 function setProductImage(imgEl, srcFromAPI) {
  const PLACEHOLDER = '/imgproductos/placeholder.png';

  // si no viene path → placeholder
  if (!srcFromAPI) { imgEl.src = PLACEHOLDER; imgEl.onerror = null; return; }

  // normaliza a ruta absoluta y fuerza .webp
  let base = String(srcFromAPI || '').trim();
  if (!/^https?:\/\//i.test(base) && !base.startsWith('/')) base = '/' + base;
  base = base.replace(/\.[^.]+$/, '') + '.webp';

  imgEl.onerror = () => { imgEl.src = PLACEHOLDER; };
  imgEl.src = base;
}

// === MENSAJE POR CATEGORÍA (público) ===
// Usa BE y normalizeForKey que ya existen en este archivo.
let __catSettingsCache = null;

// Crea/asegura el banner y lo devuelve
function ensureCategoryBanner() {
  let el = document.getElementById('category-message');
  if (!el) {
    el = document.createElement('div');
    el.id = 'category-message';
    el.className = 'cat-msg-banner';
    el.hidden = true;
    const parent = document.querySelector('header') || document.body;
    parent.insertBefore(el, parent.firstChild);
  }
  return el;
}

// Descarga el mapa público: { key : {message,is_active,updated_at} }
async function fetchCategorySettingsPublic(){
  try {
    const r = await fetch(BE + 'category-settings-public.php?_=' + Date.now(), { cache: 'no-store' });
    const rawData = await r.json() || {};
    // Normaliza las claves de las categorías al slug utilizado internamente.
    // La API devuelve un objeto cuyas claves son los nombres tal cual en la
    // base de datos.  Para que resolveCategoryKey y applyCategoryBanner
    // funcionen correctamente, convertimos esas claves a su versión
    // normalizada (slug) y preservamos el nombre original en la propiedad
    // `raw`.  Esto evita discrepancias entre mayúsculas/minúsculas y
    // espacios/acentos en las categorías.
    const normalized = {};
    for (const [rawKey, val] of Object.entries(rawData)) {
      const slug = normalizeForKey(rawKey);
      // Copiar campos y añadir el nombre original
      normalized[slug] = Object.assign({}, val, { raw: rawKey });
    }
    __catSettingsCache = normalized;
  } catch {
    __catSettingsCache = {};
  }
  return __catSettingsCache;
}

// Algunos proyectos antiguos usan "base-whisky" vs "whisky"
// Ajusta aquí si te quedan prefijos raros
const CAT_ALIAS = {
  'whisky': 'base-whisky'
  // agrega más si hiciera falta, ej: 'ron':'base-ron'
};

// Recibe una “etiqueta” (texto del botón) o una “key” y resuelve la clave real
function resolveCategoryKey(anyKeyOrLabel){
  const norm = normalizeForKey(anyKeyOrLabel);
  if (__catSettingsCache && __catSettingsCache[norm]) return norm;
  if (CAT_ALIAS[norm]) return CAT_ALIAS[norm];

  // dataset trae key; si vino etiqueta, intenta mapear etiqueta→key
  if (window.categoryLabels) {
    // ya es key
    if (window.categoryLabels[norm]) return norm;
    // etiqueta → key
    for (const [k, v] of Object.entries(window.categoryLabels)) {
      if (normalizeForKey(v) === norm) return k;
    }
  }
  return norm;
}

// Aplica (pinta/oculta) el banner según la categoría activa
function applyCategoryBanner(activeKeyOrLabel){
  const el = ensureCategoryBanner();
  if (!__catSettingsCache) return;       // aún no descargado; lo hará el init
  const key = resolveCategoryKey(activeKeyOrLabel);
  const cfg = __catSettingsCache[key];
  const txt = (cfg && Number(cfg.is_active) === 1) ? String(cfg.message || '').trim() : '';
  if (txt) { el.textContent = txt; el.hidden = false; }
  else { el.textContent = ''; el.hidden = true; }
}

// Expón por si lo llaman otros scripts
window.ensureCategoryBanner = ensureCategoryBanner;
window.fetchCategorySettingsPublic = fetchCategorySettingsPublic;
window.applyCategoryBanner = applyCategoryBanner;


  // === BACKEND HELPERS ===
  async function fetchTasaBCV() {
    try {
      const res = await fetch(BE + 'bcv.php', { cache: 'no-store' });
      const data = await res.json();
      window.tasaBCV = parseFloat(data.tasa) || 1;
    } catch { window.tasaBCV = 1; }
    const tasaEl = document.getElementById('tasa-bcv');
    if (tasaEl) tasaEl.textContent = `Tasa BCV: $ ${window.tasaBCV.toFixed(2)}`;
  }

  async function fetchProductos() {
    try {
      const res = await fetch(BE + 'products.php', { cache: 'no-store' });
      const arr = await res.json();
      const grouped = {};
      const labels = {};
      (Array.isArray(arr) ? arr : []).forEach(p => {
        const origLabel = (p.category || p.categoria || 'otros') + '';
        const key = normalizeForKey(origLabel || 'otros') || 'otros';
        if (!grouped[key]) grouped[key] = [];

        // Imagen segura
        let imgRaw = (p.img || '').toString().trim();
        let safeImg = '';
        if (!imgRaw) safeImg = '';
        else if (/^https?:\/\//i.test(imgRaw)) safeImg = imgRaw;
        else if (imgRaw.startsWith('/')) safeImg = imgRaw;
        else safeImg = '/' + imgRaw;

        const catSlug  = normalizeForKey(origLabel || 'otros');
        const nameSlug = normalizeForKey(p.name || p.nombre || '');
        if (!safeImg) safeImg = `/imgproductos/${catSlug}/${nameSlug}`;
        try { safeImg = safeImg.replace(/[^/]+$/, file => encodeURIComponent(file)); } catch {}

        // Disponibilidad inicial (si la API lo expone)
        const available =
          (p.available !== undefined) ? !!p.available :
          (p.is_active !== undefined) ? (Number(p.is_active) === 1) :
          true;

        grouped[key].push({
          ...p,
          id: p.id,
          img: safeImg,
          available: available
        });
        if (!labels[key]) labels[key] = origLabel;
      });
      window.data = grouped;
      window.categoryLabels = labels;
    } catch {
      // Si la petición al backend falla (por ejemplo, no hay servidor PHP)
      // establecemos data vacía y aplicamos las categorías predeterminadas
      window.data = {};
      // Clonamos DEFAULT_CATEGORY_LABELS en lugar de asignarlo directamente
      // para evitar modificaciones accidentales sobre la constante.
      window.categoryLabels = Object.assign({}, DEFAULT_CATEGORY_LABELS);
    }
  }

  // === Disponibilidad (siempre fresca del backend) ===
async function fetchActiveIdSet() {
  try {
    const r = await fetch(BE + 'products.php', { cache: 'no-store' });
    const arr = await r.json();
    const s = new Set();
    (Array.isArray(arr) ? arr : []).forEach(p => {
      // Determinar disponibilidad: prioriza `available`, luego `is_active`, y por defecto true
      const isAvailable =
        (p && p.available !== undefined) ? !!p.available :
        (p && p.is_active !== undefined) ? (Number(p.is_active) === 1) :
        true;
      // Solo añade el ID si el producto está disponible
      if (isAvailable && p && p.id != null) {
        s.add(String(p.id));
      }
    });
    return s;
  } catch {
    return new Set();
  }
}

  async function isProductIdActive(id) {
    const s = await fetchActiveIdSet();
    return s.has(String(id));
  }
  function markCardUnavailable(card) {
    if (!card) return;
    card.setAttribute('data-available', '0');
    card.classList.add('is-unavailable');
    const btn = card.querySelector('.select, .btn-seleccionar, [data-action="select"]');
    if (btn) btn.setAttribute('aria-disabled', 'true');
  }
  async function syncAvailabilityFromBackend() {
    const set = await fetchActiveIdSet();
    document.querySelectorAll('.card[data-product-id], .product-card[data-product-id], .producto[data-product-id]')
      .forEach(card => {
        const id = card.getAttribute('data-product-id');
        const active = set.has(String(id));
        card.setAttribute('data-available', active ? '1' : '0');
        card.classList.toggle('is-unavailable', !active);
      });
  }

  // === Destacar categoría "comida-rapida" ===
  function markComidaRapidaBlink() {
    const cont = document.getElementById('categorias-container');
    if (!cont) return;
    const btn = [...cont.querySelectorAll('button')].find(b => {
      const raw = (b.dataset && (b.dataset.categoria || b.getAttribute('data-categoria'))) || b.textContent || '';
      const key = normalizeForKey(raw);
      return key === 'comida-rapida';
    });
    if (btn && !btn.classList.contains('cat-blink')) {
      btn.classList.add('cat-blink');
      btn.setAttribute('aria-label', (btn.getAttribute('aria-label') || 'comida-rapida') + ' destacado');
    }
  }
  const catCont = document.getElementById('categorias-container');
  if (catCont) {
    markComidaRapidaBlink();
    const mo = new MutationObserver(markComidaRapidaBlink);
    mo.observe(catCont, { childList: true });
  }

  // === Usuario ===
  async function fetchUsuarioPorTelefono(tel) {
    if (!tel) return null;
    try {
      const url = BE + 'usuarios.php?telefono=' + encodeURIComponent(tel);
      const res = await fetch(url, { cache: 'no-store' });
      const data = await res.json();
      const getRawUser = (d) => {
        if (!d) return null;
        if (d.usuario) return d.usuario;
        if (d.user) return d.user;
        if (d.data && !Array.isArray(d.data) && typeof d.data === 'object') return d.data;
        if (Array.isArray(d.data) && d.data.length) return d.data[0];
        if (Array.isArray(d) && d.length) return d[0];
        return null;
      };
      const raw = getRawUser(data);
      if (!raw) return null;
      return {
        id: raw.id || null,
        nombre: raw.nombre || raw.name || '-',
        telefono: raw.telefono || raw.phone || tel,
        puntos: parseInt(raw.puntos ?? raw.coins ?? raw.puntos ?? 0) || 0,
        maxpuntos: parseInt(raw.maxpuntos ?? raw.maxCoins ?? raw.puntos ?? 0) || 0,
        avatarURL: raw.avatarURL || raw.avatarUrl || raw.avatar || '',
        esBorrachinPudiente: !!(raw.esBorrachinPudiente || raw.esPudiente || raw.maspudiente || raw.isTop)
      };
    } catch (e) {
      console.error('[fetchUsuarioPorTelefono] error:', e);
      return null;
    }
  }
  async function fetchEnviarPedido(pedido) {
    try {
      const res = await fetch(BE + 'orders.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(pedido)
      });
      return await res.json();
    } catch { return { success: false, error: 'Error de red.' }; }
  }



  // === RENDER DE PRODUCTOS ===
  function renderProducts(categoryKey = 'all') {
    const grid = document.getElementById('product-grid');
    if (!grid) return;
    grid.innerHTML = '';
    const keys = (categoryKey === 'all' || categoryKey === 'todos') ? Object.keys(window.data) : [categoryKey];
    keys.forEach(k => {
      const list = window.data[k] || [];
      list.forEach(p => {
        const priceNum = (typeof p.price === 'number') ? p.price : (parseFloat(p.price) || 0);
        const precioBs = (priceNum * window.tasaBCV).toFixed(2);

        const available =
          (p.available !== undefined) ? !!p.available :
          (p.is_active !== undefined) ? (Number(p.is_active) === 1) :
          true;

        const card = document.createElement('div');
        card.className = 'card';
        if (p.id != null) card.setAttribute('data-product-id', String(p.id));
        card.setAttribute('data-available', available ? '1' : '0');
        if (!available) card.classList.add('is-unavailable');

        const imgEl = document.createElement('img');
        imgEl.alt = (p.name || p.nombre || 'Producto');
        imgEl.loading = 'lazy';
        setProductImage(imgEl, p.img);
        card.appendChild(imgEl);

        const h3 = document.createElement('h3');
        h3.className = 'name';
        h3.textContent = p.name || p.nombre || '';
        card.appendChild(h3);

        const desc = document.createElement('p');
        desc.className = 'description';
        // Support different property names for description.  Some API endpoints
        // return the field as `description`, others use `desc` or `descripcion`.
        // Fall back gracefully in order of preference.  This ensures that
        // descriptions saved for newly added products are displayed correctly.
        desc.textContent = p.description || p.desc || p.descripcion || '';
        card.appendChild(desc);

        const priceWrap = document.createElement('p');
        priceWrap.className = 'price';
        priceWrap.dataset.price = priceNum;
        priceWrap.innerHTML = `USD ${priceNum.toFixed(2)}<br>MonedaLocal ${precioBs}`;
        card.appendChild(priceWrap);

        const qty = document.createElement('div');
        qty.className = 'quantity';
        const btnDec = document.createElement('button'); btnDec.className = 'dec'; btnDec.textContent = '−';
        const spanCount = document.createElement('span'); spanCount.className = 'count'; spanCount.textContent = '0';
        const btnInc = document.createElement('button'); btnInc.className = 'inc'; btnInc.textContent = '+';
        qty.appendChild(btnDec); qty.appendChild(spanCount); qty.appendChild(btnInc);
        card.appendChild(qty);

        const selectBtn = document.createElement('button');
        selectBtn.className = 'select btn-seleccionar';
        selectBtn.setAttribute('data-action', 'select');
        selectBtn.textContent = 'Seleccionar';
        card.appendChild(selectBtn);

        grid.appendChild(card);
      });
    });
    attachListeners();
    // sincroniza estado visual justo luego de pintar
    syncAvailabilityFromBackend();
  }
// --- MENSAJE al clicar en una card NO disponible ---
(function(){
  const grid = document.getElementById('product-grid');
  if (!grid || grid._noDispHandler) return;
  grid._noDispHandler = true;

  // Usamos captura para atrapar el click aunque haya estilos raros
grid.addEventListener('click', (ev) => {
  const card = ev.target.closest('.card, .product-card, .producto');
  if (!card) return;

  const notAvailable = card.getAttribute('data-available') === '0' || card.classList.contains('is-unavailable');
  if (!notAvailable) return;

  // Solo si tocó estos elementos:
  const onAction = ev.target.closest('.select, .btn-seleccionar, .inc, .dec, .quantity');
  if (!onAction) return;

  ev.preventDefault();
  ev.stopPropagation();
  showNeonAlert('⛔ Producto no disponible');
}, true);

})();

// ========================================================
// Botón de Administración para la página principal
// ========================================================
// Este bloque crea y gestiona un botón de administración en la página
// principal. El botón permanece oculto para usuarios no administradores
// y redirige al panel ubicado en /admin/ cuando se hace clic.
document.addEventListener('DOMContentLoaded', () => {
  const ADMIN_BUTTON_ID = 'admin-toggle';

  // Crea el botón si no existe
  function ensureAdminButton() {
    let btn = document.getElementById(ADMIN_BUTTON_ID);
    if (!btn) {
      btn = document.createElement('button');
      btn.id = ADMIN_BUTTON_ID;
      btn.className = 'admin-toggle';
      btn.textContent = '🔐 Modo Admin';
      // Estilos básicos similares al botón original
      btn.style.cssText = [
        'position:fixed',
        'bottom:16px',
        'right:16px',
        'padding:10px 14px',
        'border-radius:12px',
        'background:#111',
        'color:#0ff',
        'border:1px solid #0ff',
        'box-shadow:0 0 12px #0ff',
        'z-index:2147483647',
        'font-weight:600',
        'letter-spacing:.3px',
        'cursor:pointer',
        'display:none'
      ].join(';');
      // Al hacer clic redirige al panel admin
      btn.addEventListener('click', () => {
        window.location.href = '/admin/admin.php';
      });
      document.body.appendChild(btn);
    }
    return btn;
  }

  // Expone funciones globales para mostrar u ocultar el botón
  function revealAdminNow() {
    const btn = ensureAdminButton();
    btn.style.display = 'block';
  }
  function hideAdminNow() {
    const btn = document.getElementById(ADMIN_BUTTON_ID);
    if (btn) btn.style.display = 'none';
  }

  // Hacer disponible a otros scripts
  window.revealAdminNow = revealAdminNow;
  window.hideAdminNow = hideAdminNow;

  // Crear el botón inicialmente (oculto)
  ensureAdminButton();
});

  // === LISTENERS (incluye guardias de disponibilidad) ===
  function attachListeners() {
    document.querySelectorAll('.dec').forEach(b =>
      b.onclick = () => {
        const c = b.nextElementSibling;
        c.textContent = Math.max(0, +c.textContent - 1);
      }
    );
    document.querySelectorAll('.inc').forEach(b =>
      b.onclick = () => {
        const c = b.previousElementSibling;
        c.textContent = +c.textContent + 1;
      }
    );

    document.querySelectorAll('.select').forEach(btn => {
      btn.classList.add('btn-seleccionar');
      btn.setAttribute('data-action', 'select');

      btn.onclick = async () => {
        const card = btn.closest('.card, .product-card, .producto');
        if (!card) return;

        // 1) estado DOM
        const availAttr = card.getAttribute('data-available');
        const isAvailableDom = (availAttr == null) ? true : (availAttr !== '0');
        if (!isAvailableDom) {
          return showNeonAlert('⛔ Producto no disponible');
        }

        // 2) verificación FRESCA con backend
        const id = card.getAttribute('data-product-id');
        if (id) {
          const active = await isProductIdActive(id);
          if (!active) {
            markCardUnavailable(card);
            return showNeonAlert('⛔ Producto no disponible');
          }
        }

        // 3) agregar al carrito
        const name  = card.querySelector('.name')?.textContent || 'Producto';
        const price = +card.querySelector('.price')?.dataset.price || 0;
        const qty   = +card.querySelector('.count')?.textContent || 0;
        if (!qty) return showNeonAlert('Selecciona al menos 1');

        window.cart[name] = { id: id || null, name, price, qty };
        btn.textContent = '✓ Agregado';
        setTimeout(() => btn.textContent = 'Seleccionar', 800);
      };
    });
  }

  // Valida carrito contra backend. Devuelve array de nombres NO disponibles
  async function validarDisponibilidadCarrito() {
    const set = await fetchActiveIdSet();
    const invalid = [];
    Object.values(window.cart).forEach(item => {
      if (item.id && !set.has(String(item.id))) invalid.push(item.name);
    });
    // marca visualmente cards inválidas si están en pantalla
    document.querySelectorAll('.card[data-product-id], .product-card[data-product-id], .producto[data-product-id]').forEach(card => {
      const id = card.getAttribute('data-product-id');
      if (id && !set.has(String(id))) markCardUnavailable(card);
    });
    return invalid;
  }

  // === APP INIT ===
  document.addEventListener('DOMContentLoaded', async () => {
    await fetchTasaBCV();
    await fetchProductos();
    // Descarga los mensajes públicos por categoría.  Esto también
    // inicializa el cache interno (__catSettingsCache) con las
    // categorías configuradas por el administrador en la base de datos.
    await fetchCategorySettingsPublic();

    // Combina las categorías guardadas por el administrador (desde
    // __catSettingsCache) con las categorías detectadas en los productos.
    // Sin esta mezcla, el listado de categorías y sus mensajes sólo
    // incluiría aquellas que provienen del Excel de productos.  Al
    // fusionar ambas fuentes mantenemos visibles las categorías que
    // hayan sido creadas o modificadas desde el panel de administración
    // aunque no estén asociadas a un producto en el inventario.
    try {
      if (typeof __catSettingsCache === 'object' && __catSettingsCache) {
        Object.keys(__catSettingsCache).forEach(rawCat => {
          const slug = normalizeForKey(rawCat);
          if (!window.categoryLabels || typeof window.categoryLabels !== 'object') {
            window.categoryLabels = {};
          }
          if (!Object.prototype.hasOwnProperty.call(window.categoryLabels, slug)) {
            // Guardamos la etiqueta original (no normalizada) como valor de la categoría
            window.categoryLabels[slug] = rawCat;
          }
        });
      }
    } catch (e) {
      // Silencioso: si falla, simplemente no añade categorías adicionales
    }

    // Aplica inicialmente el banner de mensaje sin mostrar nada para la
    // categoría "todos".  Esto se ejecuta después de la mezcla de
    // categorías para asegurar que los mensajes estén disponibles.
    applyCategoryBanner('all');

    // Asegura que window.categoryLabels contenga como valor el nombre
    // original (raw) de cada categoría almacenada en __catSettingsCache.  El
    // objeto __catSettingsCache utiliza claves normalizadas (slug) y
    // contiene una propiedad `raw` con el nombre real de la categoría.  Si
    // no existe dicha propiedad, se usa la clave misma.  Esto evita que
    // categorías creadas desde el admin pierdan su capitalización original
    // y garantiza que se puedan recuperar los mensajes correspondientes.
    if (typeof __catSettingsCache === 'object' && __catSettingsCache) {
      if (!window.categoryLabels || typeof window.categoryLabels !== 'object') {
        window.categoryLabels = {};
      }
      Object.keys(__catSettingsCache).forEach(slugKey => {
        const rec = __catSettingsCache[slugKey];
        const rawName = (rec && rec.raw) ? rec.raw : slugKey;
        window.categoryLabels[slugKey] = rawName;
      });
    }

    // Generamos dinámicamente los botones de categorías para la portada.
    // Anteriormente la funcionalidad de categorías vivía en admin‑loader.js, por lo que
    // no se ejecutaba en la página principal. Si generateCategoryButtons no
    // existe aún en esta página, creamos nuestra propia versión.  Esto
    // garantiza que las categorías se muestren correctamente a todos los
    // visitantes una vez que fetchProductos haya cargado window.categoryLabels.
    if (typeof window.generateCategoryButtons !== 'function') {
      window.generateCategoryButtons = function generateCategoryButtons() {
        const contenedor = document.getElementById('categorias-container');
        if (!contenedor) return;
        const labels = window.categoryLabels || {};
        const keys   = Object.keys(labels);
        // Si aún no hay categorías, reintenta brevemente un par de veces.
        if (!keys.length) {
          if (!generateCategoryButtons._tries) generateCategoryButtons._tries = 0;
          if (generateCategoryButtons._tries++ < 20) {
            setTimeout(generateCategoryButtons, 300);
          }
          return;
        }
        // Limpia cualquier botón anterior.
        contenedor.innerHTML = '';
        // Botón para mostrar todos los productos.
        const btnAll = document.createElement('button');
        btnAll.className = 'categoria-btn cat-btn cat-active';
        btnAll.textContent = 'Todos';
        btnAll.dataset.categoria = 'all';
        contenedor.appendChild(btnAll);
        // Crea un botón por cada categoría ordenada alfabéticamente por el
        // nombre presentado al usuario.
        keys.sort((a, b) => (labels[a] || '').localeCompare(labels[b] || ''))
          .forEach(key => {
            const btn = document.createElement('button');
            btn.className = 'categoria-btn cat-btn';
            btn.textContent = labels[key] || key;
            btn.dataset.categoria = key;
            contenedor.appendChild(btn);
          });
        // Si existen funciones para el banner de categoría en el ámbito global,
        // utilízalas para que el mensaje por categoría siga funcionando.
        if (typeof window.ensureCategoryBanner === 'function') {
          try { window.ensureCategoryBanner(); } catch { /* ignore */ }
        }
        if (typeof window.fetchCategorySettingsPublic === 'function' && typeof window.applyCategoryBanner === 'function') {
          try { window.fetchCategorySettingsPublic().then(window.applyCategoryBanner); } catch { /* ignore */ }
        }
      };
    }
    // Ejecuta la función de generación de categorías
    if (typeof window.generateCategoryButtons === 'function') {
      window.generateCategoryButtons();
    }

    const contenedorCategorias = document.getElementById('categorias-container');
    if (contenedorCategorias) {
      contenedorCategorias.addEventListener('click', (e) => {
        const btn = e.target.closest('button.cat-btn');
        if (!btn) return;

        contenedorCategorias.querySelectorAll('button.cat-btn').forEach(b => b.classList.remove('cat-active','active'));
        btn.classList.add('cat-active');

        const raw = btn.dataset.categoria || btn.textContent || 'all';
        const key = normalizeForKey(raw);
        renderProducts((key === 'all' || key === 'todos') ? 'all' : key);
        applyCategoryBanner(raw);   // ← PINTA el banner para esa categoría

      });
    }
    renderProducts('all');

    // Poll ligero para reflejar cambios del admin sin recargar
    setInterval(syncAvailabilityFromBackend, 12000);

    // Logo animado (igual que antes)
    (function(){
      const el = document.querySelector('.logo-text');
      if(!el) return;
      const groupSize = 3;
      const paletteClasses = ['g0','g1','g2'];
      const text = el.textContent.trim();
      el.setAttribute('aria-label', text);
      el.textContent = '';
      for(let i=0, g=0; i<text.length; i += groupSize, g++){
        const chunk = text.slice(i, i+groupSize);
        const span = document.createElement('span');
        span.className = paletteClasses[g % paletteClasses.length];
        span.textContent = chunk;
        el.appendChild(span);
      }
    })();

    // === CARRITO / CHECKOUT ===
    const checkoutBtn  = document.getElementById('checkout');
    const summary      = document.getElementById('modal-summary');
    const summaryList  = document.getElementById('summary-list');
    const summaryTotal = document.getElementById('summary-total');
    const confirmBtn   = document.getElementById('confirm-pea');

    if (checkoutBtn) checkoutBtn.onclick = async () => {
      if (!Object.keys(window.cart).length) return showNeonAlert('Elije cantidad y selecciona😵‍💫');

      // VALIDACIÓN de disponibilidad antes de abrir resumen
      const invalid = await validarDisponibilidadCarrito();
      if (invalid.length) {
        showNeonAlert('⛔ Producto no disponible: ' + invalid.join(', '));
        return;
      }

      const tel = sessionStorage.getItem('loggedPhone');
      if (!tel) {
        const modal = document.getElementById('login-modal');
        if (modal) modal.style.display = 'flex';
        return;
      }
      mostrarCargando('Preparando resumen...');
      await openSummary();
      ocultarCargando();
    };

    async function generarResumen() {
      if (!summaryList || !summaryTotal) return;

      // VALIDACIÓN por si el estado cambió entre click y aquí
      const invalid = await validarDisponibilidadCarrito();
      if (invalid.length) {
        showNeonAlert('⛔ Producto no disponible: ' + invalid.join(', '));
        return;
      }

      summaryList.innerHTML = '';
      let sumUsd = 0;
      Object.values(window.cart).forEach(item => {
        const sub = item.price * item.qty;
        sumUsd += sub;
        summaryList.innerHTML += `
          <li>
            ${item.qty}× ${item.name} = $${sub.toFixed(2)}
            <button class="remove-summary-item" data-name="${item.name}">🗑️</button>
          </li>`;
      });
      const code = 'Cod-' + Date.now();
      const $ = (sumUsd * window.tasaBCV).toFixed(2);
      summaryTotal.innerHTML = `
        <p>Total: $${sumUsd.toFixed(2)}</p>
        <p>Total Monedalocal: ${$}</p>
        <small>Código: ${code}</small>`;
      if (confirmBtn) confirmBtn.dataset.code = code;

      document.querySelectorAll('.remove-summary-item').forEach(btn => {
        btn.onclick = () => {
          delete window.cart[btn.dataset.name];
          generarResumen();
        };
      });
    }

    async function openSummary() {
      await generarResumen();
      const tel = sessionStorage.getItem('loggedPhone');
      const user = await fetchUsuarioPorTelefono(tel);
      insertUserInfo(user);
      if (summary) summary.style.display = 'flex';
    }

    if (summary) {
      const closeSummary = summary.querySelector('.close-modal');
      if (closeSummary) closeSummary.onclick = () => summary.style.display = 'none';
      summary.addEventListener('click', e => { if (e.target === summary) summary.style.display = 'none'; });
    }

    if (confirmBtn) {
      let enviando = false;
      confirmBtn.onclick = async () => {
        if (enviando) return;
        if (!Object.keys(window.cart).length) return showNeonAlert('🛒 Selecciona Producto');

        // VALIDACIÓN FINAL: bloquea si hay no disponibles
        const invalid = await validarDisponibilidadCarrito();
        if (invalid.length) {
          showNeonAlert('⛔ Producto no disponible: ' + invalid.join(', '));
          return;
        }

        const desp = document.querySelector('input[name="despacho"]:checked');
        const met  = document.querySelector('input[name="metodo"]:checked');
        if (!desp || !met) return showNeonAlert('❗ Elige despacho y método de pago');

        enviando = true;
        confirmBtn.disabled = true;
        const oldText = confirmBtn.textContent;
        confirmBtn.textContent = 'Procesando...';
        mostrarCargando();

        try {
          let sumUsd = 0;
          Object.values(window.cart).forEach(i => sumUsd += i.price * i.qty);
          const tel = sessionStorage.getItem('loggedPhone');
          const user = await fetchUsuarioPorTelefono(tel);
          const code = confirmBtn.dataset.code;

          const pedido = {
            id: code,
            nombre: user ? user.nombre : 'Anon',
            telefono: tel,
            montoUsd: sumUsd,
            despacho: desp.value,
            metodo: met.value,
            detalle: Object.values(window.cart)
          };
          const res = await fetchEnviarPedido(pedido);
          if (res.success) {
            showNeonAlert('✅ Pedido enviado, abriendo WhatsApp...');

            const bsTotalStr  = (sumUsd * window.tasaBCV).toLocaleString('es-VE', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
            const usdTotalStr = Number(sumUsd).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });

            const detalleLines = Object.values(window.cart)
              .map(i => `• ${i.qty}× ${i.name} — $${(i.price * i.qty).toFixed(2)}`)
              .join('\n');

            const waMsg =
              `*¡Hola!* Me gustaría hacer un pedido:\n\n` +
              `${detalleLines}\n\n` +
              `*Total:* $${usdTotalStr} / $ ${bsTotalStr}\n` +
              `*🧾 Orden:* ${code}\n` +
              `*📦 Despacho:* ${desp.value}\n` +
              `*💳 Método de Pago:* ${met.value}\n` +
              `*👤 Cliente:* ${user ? user.nombre : 'Anon'}\n` +
              `*🪙 puntos acumulados:* ${user ? (user.puntos || 0) : 0}\n\n` +
              `¡Muchas gracias!`;

            const urlWA = `${WA_TARGET}?text=${encodeURIComponent(waMsg)}`;

            let waWin = null;
            try { waWin = window.open(urlWA, '_blank'); } catch {}
            if (!waWin || waWin.closed || typeof waWin.closed === 'undefined') {
              const a = document.createElement('a');
              a.href = urlWA; a.target = '_blank'; a.rel = 'noopener';
              document.body.appendChild(a);
              a.click();
              a.remove();
              setTimeout(() => { try { window.location.href = urlWA; } catch {} }, 100);
            }

            if (summary) summary.style.display = 'none';
            window.cart = {};
            document.querySelectorAll('.count').forEach(s => s.textContent = '0');
            document.querySelectorAll('.select').forEach(b => b.textContent = 'Seleccionar');

          } else {
            showNeonAlert(res.error || 'No se pudo enviar el pedido.');
          }
        } catch (err) {
          console.error(err);
          showNeonAlert('❌ Error procesando pedido.');
        } finally {
          enviando = false;
          confirmBtn.disabled = false;
          confirmBtn.textContent = oldText;
          ocultarCargando();
        }
      };
    }

    function insertUserInfo(user) {
      const cont = document.getElementById('summary-user');
      cont && (cont.innerHTML = `<p class="user-info">Cliente: ${user ? user.nombre : '—'} — Coins: ${user ? user.puntos || 0 : 0}</p>`);
    }

    // === LOGIN / SESIÓN (sin cambios funcionales) ===
    let loggedPhone = sessionStorage.getItem('loggedPhone') || null;
    function updateSessionUI(user) {
      window.currentUser = user || null;
      const greetingEl = document.getElementById('user-greeting');
      const loginBtn   = document.getElementById('btn-login');
      const logoutBtn  = document.getElementById('btn-logout');

      if (user) {
        if (greetingEl) {
          greetingEl.style.display = 'inline';
          greetingEl.textContent = `Hola ${user.nombre}`;
        }
        if (loginBtn) loginBtn.style.display = 'none';
        if (logoutBtn) logoutBtn.style.display = 'inline';
      } else {
        if (greetingEl) greetingEl.style.display = 'none';
        if (loginBtn) loginBtn.style.display = 'inline';
        if (logoutBtn) logoutBtn.style.display = 'none';
      }

      // Mostrar u ocultar el botón de administración según el teléfono del usuario
      try {
        // window.ADMIN_USER se define en admin-loader.js; si no existe, usa el valor por defecto
        const adminPhone = (window.ADMIN_USER || '04121234567').trim();
        if (user && String(user.telefono || '').trim() === adminPhone) {
          // Usuario administrador: mostrar el botón
          if (typeof window.revealAdminNow === 'function') window.revealAdminNow();
        } else {
          // Usuario no administrador o sin sesión: ocultar el botón y resetear estado admin
          if (typeof window.hideAdminNow === 'function') window.hideAdminNow();
          try { window.__IS_ADMIN__ = false; } catch {}
        }
      } catch (e) {
        console.warn('No se pudo actualizar el estado del botón admin:', e);
      }
    }
    window.updateSessionUI = updateSessionUI;

    async function confirmLogin() {
      const phoneEl = document.getElementById('login-phone');
      const errEl   = document.getElementById('login-error');
      if (!phoneEl || !errEl) return;

      const tel = phoneEl.value.trim();
      if (!/^\d{11}$/.test(tel)) {
        errEl.textContent = 'ingrese 11 dígitos numéricos';
        errEl.style.display = 'block';
        return;
      }

      try {
        const response = await fetch(BE + 'login.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ telefono: tel })
        });
        const data = await response.json();
        if (data.success && data.usuario) {
          sessionStorage.setItem('loggedPhone', tel);
          loggedPhone = tel;
          updateSessionUI(data.usuario);
          closeLoginModal();
        } else {
          errEl.textContent = 'Número no registrado. Redirigiendo a registro…';
          errEl.style.display = 'block';
          setTimeout(() => {
            closeLoginModal();
            const reg = document.getElementById('modal-registro');
            if (reg) reg.style.display = 'flex';
          }, 1500);
        }
      } catch (error) {
        console.error('Error en login:', error);
        errEl.textContent = 'Error de conexión';
        errEl.style.display = 'block';
      }
    }
    function closeLoginModal() {
      const modal = document.getElementById('login-modal');
      if (modal) modal.style.display = 'none';
      const err = document.getElementById('login-error');
      if (err) err.style.display = 'none';
    }
    const loginBtnEl   = document.getElementById('btn-login');
    const logoutBtnEl  = document.getElementById('btn-logout');
    if (loginBtnEl)  loginBtnEl.onclick = () => { document.getElementById('login-modal').style.display = 'flex'; };
    if (logoutBtnEl) logoutBtnEl.onclick = async () => {
      if (!loggedPhone) return;
      if (!confirm('¿Deseas cerrar sesión?')) return;
      sessionStorage.removeItem('loggedPhone');
      loggedPhone = null;
      updateSessionUI(null);
      showNeonAlert('👋 Sesión cerrada');
    };
    const loginCloseEl = document.querySelector('#login-modal .login-close');
    if (loginCloseEl) loginCloseEl.onclick = closeLoginModal;
    const loginModalEl = document.getElementById('login-modal');
    if (loginModalEl) loginModalEl.addEventListener('click', e => { if (e.target === loginModalEl) closeLoginModal(); });
    const loginConfirmEl = document.getElementById('login-confirm');
    if (loginConfirmEl) loginConfirmEl.onclick = async () => {
      loginConfirmEl.disabled = true;
      const old = loginConfirmEl.textContent;
      loginConfirmEl.textContent = 'Procesando…';
      try { await confirmLogin(); } finally { loginConfirmEl.textContent = old; loginConfirmEl.disabled = false; }
    };

    const consultEl = document.getElementById('btn-consultar-coins');
    if (consultEl) consultEl.onclick = async () => {
      const tel = sessionStorage.getItem('loggedPhone');
      if (!tel) { const lm = document.getElementById('login-modal'); if (lm) lm.style.display = 'flex'; return; }
      if (typeof mostrarCarnetUsuario === 'function') mostrarCarnetUsuario(tel);
    };

    if (loggedPhone) {
      const user = await fetchUsuarioPorTelefono(loggedPhone);
      updateSessionUI(user);
    }
  });

})(); // IIFE end

// === PROMO VIDEO (igual) ===
document.addEventListener('DOMContentLoaded', async () => {
  const modal    = document.getElementById('promoVideoModal');
  const closeBtn = document.getElementById('closePromoVideo');
  const video    = document.getElementById('promoVideo');
  if (!modal || !closeBtn || !video) return;

  const BE2 = (window.BACKEND || '/api/').replace(/\/?$/, '/');

  let src = '';
  try {
    const r = await fetch(BE2 + 'promo_config.php', { cache: 'no-store' });
    const cfg = await r.json();
    if (cfg && cfg.active && cfg.src) src = cfg.src;
  } catch {}
  const APP_VERSION = '2025-08-14-3';
  if (!src) src = `/videopromo.mp4?v=${APP_VERSION}`;

  video.src = src;
  modal.style.display = 'flex';
  closeBtn.style.display = 'none';

  try { await video.play(); } catch (e) { console.warn('Autoplay falló:', e); }

  let closable = false;
  setTimeout(() => { closeBtn.style.display = 'flex'; closable = true; }, 3000);

  closeBtn.addEventListener('click', (e) => {
    if (!closable) return;
    e.preventDefault(); e.stopPropagation();
    modal.style.display = 'none';
    try { video.pause(); video.currentTime = 0; } catch {}
  });

  video.addEventListener('error', () => {
    console.error('No se pudo cargar el video:', src, video.error);
    modal.style.display = 'none';
  });
});

// ===== Guard de selección por disponibilidad (captura global) =====
(function(){
  const SELECTORS = '.btn-seleccionar, .btn-agregar, .add-to-cart, .select, [data-action="select"]';
  document.addEventListener('click', function(ev){
    const btn = ev.target.closest(SELECTORS);
    if (!btn) return;
    const card = btn.closest('[data-product-id], .product-card, .card, .producto');
    if (!card) return;
    const availAttr = card.getAttribute('data-available');
    const isAvailable = (availAttr == null) ? true : (availAttr !== '0');
    if (!isAvailable) {
      ev.preventDefault();
      ev.stopImmediatePropagation();
      if (typeof window.showNeonAlert === 'function') {
        window.showNeonAlert('⛔ Producto no disponible');
      } else {
        alert('⛔ Producto no disponible');
      }
    }
  }, true);
})();