<?php
// backend/guard.php
// Guardas reutilizables para proteger endpoints del panel admin.
// - require_admin(): exige sesión admin y (opcional) allowlist por IP
// - require_csrf_for_write(): exige token CSRF en operaciones de escritura

declare(strict_types=1);

require_once __DIR__ . '/config.php';

// Alineamos cookies de sesión con auth.php
if (session_status() !== PHP_SESSION_ACTIVE) {
  session_set_cookie_params([
    'lifetime' => 0,
    'path'     => '/',
    'secure'   => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'),
    'httponly' => true,
    'samesite' => 'Lax',
  ]);
  session_start();
}

// --- Helpers JSON de error (no rompas la salida con HTML) ---
function guard_respond(array $payload, int $code = 403): void {
  if (!headers_sent()) {
    header('Content-Type: application/json; charset=utf-8');
    http_response_code($code);
    header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
    header('Pragma: no-cache');
    header('Expires: 0');
  }
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

// IP real considerando CDN/proxy (Cloudflare / X-Forwarded-For)
function guard_client_ip(): string {
  $ip = $_SERVER['HTTP_CF_CONNECTING_IP']
     ?? $_SERVER['HTTP_X_FORWARDED_FOR']
     ?? $_SERVER['REMOTE_ADDR']
     ?? '';
  if (strpos($ip, ',') !== false) {
    $ip = trim(explode(',', $ip)[0]);
  }
  return $ip;
}

function guard_is_admin(): bool {
  return !empty($_SESSION['is_admin']);
}

/**
 * Exige sesión de admin y (si ALLOWED_IPS no está vacía) que la IP esté permitida.
 * Responde JSON y sale si falla.
 */
function require_admin(): void {
  if (!guard_is_admin()) {
    guard_respond(['success' => false, 'error' => 'Solo administradores'], 401);
  }

  // Allowlist de IPs solo si está configurada y no vacía
  if (defined('ALLOWED_IPS') && is_array(ALLOWED_IPS) && !empty(ALLOWED_IPS)) {
    $clientIp = guard_client_ip();
    if (!in_array($clientIp, ALLOWED_IPS, true)) {
      guard_respond(['success' => false, 'error' => 'Acceso no permitido por IP'], 403);
    }
  }
}

/**
 * Exige CSRF para métodos de escritura (POST/PUT/PATCH/DELETE).
 * Toma token de header 'X-CSRF-Token' o del campo POST 'csrf'.
 */
function require_csrf_for_write(): void {
  $method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
  if (!in_array($method, ['POST','PUT','PATCH','DELETE'], true)) {
    return; // solo aplica a escritura
  }

  $sessionToken = $_SESSION['csrf'] ?? '';
  $headerToken  = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
  $postToken    = $_POST['csrf'] ?? '';

  $provided = $headerToken ?: $postToken;

  if (!$sessionToken || !$provided || !hash_equals($sessionToken, $provided)) {
    guard_respond(['success' => false, 'error' => 'CSRF inválido o ausente'], 419);
  }
}
