// admin-loader.js — Versión Backend (PHP/MySQL/FETCH)
// Reemplaza TODAS las referencias a localStorage por fetch/backend

document.addEventListener('DOMContentLoaded', () => {
  // =============================
  // Config y helpers backend
  // =============================
const BE        = (window.BACKEND || '/api/').replace(/\/?$/, '/');         // lecturas públicas + login
 const ADMIN_BE  = (window.BACKEND_AUTH || '/backend/').replace(/\/?$/, '/'); // escrituras/admin

 const CAT_ADMIN  = ADMIN_BE + 'category-settings.php'; // ← SOLO este va al backend
 const PUBLIC_BASES = [
   BE,           // normalmente /api/
   '/backend/',  // fallback a backend (category-settings-public.php)
   '/api/',
   '/'
 ];
  // Usuario administrador predeterminado (debe coincidir con ADMIN_USER en config.php)
  const ADMIN_USER = '04121234567';   // debe coincidir con config.php
  window.ADMIN_USER = ADMIN_USER;

  // Peticiones autenticadas al backend (CSRF + cookies de sesión)
  function adminFetch(url, opts = {}) {
    const csrf = window.__CSRF__ || '';
    const baseHeaders = opts.headers || {};
    const headers = csrf ? { ...baseHeaders, 'X-CSRF-Token': csrf } : { ...baseHeaders };
    const options = { ...opts, headers, credentials: 'include' };
    return fetch(url, options);
  }
  window.adminFetchGlobal = adminFetch;

  // Normalizador simple (coincide con data-categoria)
  const slugify = (s) => String(s || '')
    .normalize('NFD').replace(/[\u0300-\u036f]/g, '')
    .toLowerCase().trim()
    .replace(/\s+/g, '-')
    .replace(/[^a-z0-9\-]+/g, '')
    .replace(/-+/g, '-') || 'otros';

  // =============================
  // Utilidades UI
  // =============================
  const IMAGE_UPLOAD_BASE = '/uploads/';
  const PLACEHOLDER_IMG   = IMAGE_UPLOAD_BASE + 'placeholder.png';

// Reemplaza la función showNeonAlert actual por esta versión completa:
(function ensureNeonAlert(){
  // Evita doble init si el script se inyecta más de una vez
  if (window.__NEON_READY__) return; 
  window.__NEON_READY__ = true;

  // Asegura el contenedor
  let el = document.getElementById('neon-alert');
  if (!el) {
    el = document.createElement('div');
    el.id = 'neon-alert';
    el.className = 'neon-alert';
    document.body.appendChild(el);
  }
  (function(){
      const openAvail = document.getElementById('open-availability-modal');
      const availModal = document.getElementById('availability-modal');
      const availClose = document.getElementById('availability-close');
      if (openAvail && availModal && availClose) {
        openAvail.addEventListener('click', () => { availModal.style.display = 'block'; });
        availClose.addEventListener('click', () => { availModal.style.display = 'none'; });
        window.addEventListener('keydown', (e) => {
          if (e.key === 'Escape' && availModal.style.display !== 'none') availModal.style.display = 'none';
        });
      }
    })();
  // Implementación real (con animación)
  function neon(msg, opts = {}) {
    const text = (msg == null) ? '' : String(msg);
    const duration = Number(opts.duration) || 2600; // ms

    el.textContent = text;
    el.style.display = 'flex';

    // Reinicia animación
    el.classList.remove('neon-show');
    // fuerza reflow
    void el.offsetWidth;
    el.classList.add('neon-show');

    // Cierra automáticamente
    clearTimeout(neon._t);
    neon._t = setTimeout(() => {
      el.classList.remove('neon-show');
      el.style.display = 'none';
    }, duration);
  }

  // Expone global y alias local para compatibilidad
  window.showNeonAlert = neon;
})();


  // =============================
  // Referencias DOM
  // =============================
  const adminLoginModal  = document.getElementById('admin-login-modal');
  const adminPassword    = document.getElementById('admin-password');
  const adminLoginBtn    = document.getElementById('admin-login-btn');
  const uploadModal      = document.getElementById('admin-panel');
  const closeUpload      = uploadModal ? uploadModal.querySelector('.upload-close') : null;

  const btnVerRegistros     = document.getElementById('btn-ver-registros-admin');
  const modalUsuarios       = document.getElementById('modal-usuarios-registrados');
  const cerrarModalUsuarios = document.getElementById('cerrar-modal-usuarios');

  const tabUsuarios      = document.getElementById('tab-usuarios');
  const tabPagos         = document.getElementById('tab-pagos');
  const tabSolicitudes   = document.getElementById('tab-solicitudes');

  const contenedorCategorias = document.getElementById('categorias-container') || document.getElementById('category-buttons');
  const contentUsuarios    = document.getElementById('content-usuarios');
  const contentPagos       = document.getElementById('content-pagos');
  const contentSolicitudes = document.getElementById('content-solicitudes');
  const tbodyUsuarios    = document.getElementById('tbody-usuarios-registrados');
  const tbodyPagos       = document.querySelector('#content-pagos table tbody');
  const tbodySolicitudes = document.getElementById('tbody-solicitudes-canje');
  const excelInput       = document.getElementById('excel-file');
  const manualTasaInput  = document.getElementById('manual-tasa');
  const guardarTasaBtn   = document.getElementById('guardar-tasa');

  // =======================
  // MODO ADMIN (botón + desbloqueos)
  // =======================
  function ensureAdminButton() {
    let btn = document.getElementById('admin-toggle');
    if (!btn) {
      btn = document.createElement('button');
      btn.id = 'admin-toggle';
      btn.type = 'button';
      btn.textContent = 'Admin';
      btn.style.cssText = [
        'position:fixed','bottom:16px','right:16px','padding:10px 14px',
        'border-radius:12px','background:#111','color:#0ff',
        'border:1px solid #0ff','box-shadow:0 0 12px #0ff',
        'z-index:2147483647','display:none','visibility:hidden','opacity:0',
        'font-weight:600','letter-spacing:.3px','cursor:pointer'
      ].join(';');
      document.body.appendChild(btn);
    }
    // Click: si ya eres admin abre panel, si no abre login
    btn.onclick = () => {
      if (window.__IS_ADMIN__) openAdminPanel();
      else if (adminLoginModal) adminLoginModal.style.display = 'flex';
    };
    return btn;
  }

  function revealAdmin(){
    const el = ensureAdminButton();
    el.hidden = false;
    if (el.classList) el.classList.remove('hidden');
    el.style.setProperty('display','block','important');
    el.style.opacity = '1';
    el.style.visibility = 'visible';
  }
  window.revealAdminNow = revealAdmin;

  // Watchdog: si el botón desaparece por CSS/JS, lo re-creamos
  // Nota: el botón de administración se crea siempre pero permanece oculto hasta
  // que un usuario con credenciales de administrador inicia sesión.
  setInterval(() => {
    if (!document.getElementById('admin-toggle')) ensureAdminButton();
  }, 1500);

  function hideAdmin() {
    const btn = document.getElementById('admin-toggle');
    if (!btn) return;
    btn.hidden = true;
    // Establecer display con !important para anular cualquier valor previo
    btn.style.setProperty('display', 'none', 'important');
    btn.style.opacity = '0';
    btn.style.visibility = 'hidden';
  }
  window.hideAdminNow = hideAdmin;

  function openAdminPanel() {
    if (!uploadModal) return;
    uploadModal.style.display = 'flex';   // tu CSS usa flex
    uploadModal.focus?.();
  }
  window.openAdminPanel = openAdminPanel;
  window.__IS_ADMIN__ = window.__IS_ADMIN__ || false;

  // Cerrar modal de login (X)
  (function(){
    const cerrarLogin = document.getElementById('cerrar-login-admin');
    if (cerrarLogin) cerrarLogin.addEventListener('click', () => {
      if (adminLoginModal) adminLoginModal.style.display = 'none';
    });
  })();

  // Forzar botón inicial invisible pero presente
  ensureAdminButton();

  // =======================
  // Login admin (usuario + contraseña)
  // =======================
  if (adminLoginBtn) adminLoginBtn.addEventListener('click', async () => {
    try {
      const passVal = (adminPassword && adminPassword.value || '').trim();
      if (!passVal) throw new Error('Coloca tu clave');

      const params = new URLSearchParams({ user: window.ADMIN_USER || '', pass: passVal });

      const res = await adminFetch(ADMIN_BE + 'auth.php?action=login', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: params
      });

      const ct = (res.headers.get('content-type') || '').toLowerCase();
      const payload = ct.includes('application/json') ? await res.json() : { success:false, error: await res.text() };

      if (!res.ok || !payload.success) throw new Error(payload.error || `Login fallido (${res.status})`);

      window.__CSRF__ = payload.csrf;
      window.__IS_ADMIN__ = !!payload.is_admin;

      if (adminLoginModal) adminLoginModal.style.display = 'none';
showNeonAlert('✅ Bienvenido, admin');
      // Abre panel y corre inits
      openAdminPanel();
      cargarUsuariosEnTabla?.();
      cargarPedidosPendientes?.();
      renderizarSolicitudesCanje?.();
      initAvailabilityAdmin?.();
      initFinanzasAdmin?.();
      initAddProductAdmin?.();
      initCatMessageAdmin?.();

    } catch (err) {
      showNeonAlert(`❌ No se pudo iniciar sesión: ${err.message || err}`);
    }
  });

  if (closeUpload) closeUpload.addEventListener('click', () => {
    uploadModal.style.display = 'none';
    const el = document.getElementById('admin-toggle');
    if (el) el.style.display = 'block';
  });

  // =============================
  // Excel: Cargar productos y subir al backend
  // =============================
  if (excelInput) {
    excelInput.addEventListener('change', async function () {
      const file = this.files && this.files[0];
      if (!file) return showNeonAlert('⚠️ No seleccionaste archivo.');

      try {
        const arrayBuffer = await file.arrayBuffer();
        const wb    = XLSX.read(new Uint8Array(arrayBuffer), { type: 'array' });
        const sheet = wb.Sheets[wb.SheetNames[0]];
        const rows  = XLSX.utils.sheet_to_json(sheet, { defval: '' });
        if (!rows.length) throw new Error('Excel vacío.');

        const normalizeKey = k => String(k).toLowerCase().trim();
        let productsParsed = rows.map((r) => {
          const obj = {};
          Object.keys(r).forEach(k => obj[normalizeKey(k)] = r[k]);
          return {
            category: obj['categoria'] || obj['category'] || 'otros',
            name:     obj['nombre']    || obj['name'],
            desc:     obj['descripcion'] || obj['desc'] || '',
            price:    parseFloat(String(obj['precio'] || obj['price'] || '').replace(',', '.')) || 0,
            img:      obj['img'] || obj['imagen'] || ''
          };
        }).filter(p => p.name && p.price);

        const basename = (p) => String(p || '').split(/[\\/]/).pop();
        productsParsed = productsParsed.map(p => {
          const slug = slugify(p.category);
          const base = basename(p.img);
          if (!base) { p.img = ''; return p; }
          const baseSinExt = base.replace(/\.[^.]+$/, '');
          const finalName  = baseSinExt + '.webp';
          p.img = `/imgproductos/${slug}/${finalName}`;
          return p;
        });

        const res = await adminFetch(ADMIN_BE + 'upload-products.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ products: productsParsed })
        });
        const data = await res.json();

        if (data.success) {
          showNeonAlert('✅ Productos actualizados.');
          if (uploadModal) uploadModal.style.display = 'none';
        } else {
          showNeonAlert(data.error || 'Error al subir productos.');
        }
      } catch (err) {
        console.error(err);
        showNeonAlert(`❌ Error: ${err.message || err}`);
      }
    });
  }

  // =============================
  // Botones de categoría (desde productos del Excel)
  // =============================
  function generateCategoryButtons() {
    if (!contenedorCategorias) return;

    const labels = (window.categoryLabels || {});
    const keys = Object.keys(labels);

    if (!keys.length) {
      if (!generateCategoryButtons._tries) generateCategoryButtons._tries = 0;
      if (generateCategoryButtons._tries++ < 20) {
        setTimeout(generateCategoryButtons, 300);
      }
      return;
    }

    contenedorCategorias.innerHTML = '';

    const btnAll = document.createElement('button');
    btnAll.className = 'categoria-btn cat-btn cat-active';
    btnAll.textContent = 'Todos';
    btnAll.dataset.categoria = 'all';
    contenedorCategorias.appendChild(btnAll);

    keys
      .sort((a, b) => (labels[a] || '').localeCompare(labels[b] || ''))
      .forEach(key => {
        const btn = document.createElement('button');
        btn.className = 'categoria-btn cat-btn';
        btn.textContent = labels[key] || key;
        btn.dataset.categoria = key;
        contenedorCategorias.appendChild(btn);
      });

    // Asegurar banner debajo de los botones la primera vez
    ensureCategoryBanner();
    // Pintar con la categoría inicial
    fetchCategorySettingsPublic().then(applyCategoryBanner);
  }
  window.generateCategoryButtons = generateCategoryButtons;

  // =============================
  // Usuarios, Pedidos y Solicitudes
  // =============================
  // Abrir el modal de Registros desde el botón del panel
if (btnVerRegistros && modalUsuarios) {
  btnVerRegistros.addEventListener('click', () => {
    // Mostrar modal
    modalUsuarios.style.display = 'flex';

    // Activar pestaña "Usuarios" por defecto
    if (typeof tabUsuarios?.click === 'function') {
      tabUsuarios.click();                 // usa los listeners ya definidos para tabs
    } else {
      // Fallback: mostrar/ocultar secciones manualmente y cargar datos
      contentUsuarios && (contentUsuarios.style.display = 'block');
      contentPagos && (contentPagos.style.display = 'none');
      contentSolicitudes && (contentSolicitudes.style.display = 'none');
      cargarUsuariosEnTabla?.();
    }
  });
}

// Cerrar modal (X)
if (cerrarModalUsuarios && modalUsuarios) {
  cerrarModalUsuarios.addEventListener('click', () => {
    modalUsuarios.style.display = 'none';
  });
}

// (Opcional) Cerrar con ESC
document.addEventListener('keydown', (e) => {
  if (e.key === 'Escape' && modalUsuarios && modalUsuarios.style.display === 'flex') {
    modalUsuarios.style.display = 'none';
  }
});

  async function cargarUsuariosEnTabla() {
    if (!tbodyUsuarios) return;
    tbodyUsuarios.innerHTML = '';
    let usuarios = [];
    try {
      const res = await adminFetch(ADMIN_BE + 'admin.php?tipo=usuarios');
      usuarios = await res.json();
    } catch (e) { usuarios = []; }
    if (!Array.isArray(usuarios) || usuarios.length === 0) {
      const tr = document.createElement('tr');
      const td = document.createElement('td');
      td.colSpan = 4;
      td.textContent = 'No hay usuarios registrados.';
      tr.appendChild(td);
      tbodyUsuarios.appendChild(tr);
      return;
    }
    usuarios.forEach(u => {
      const tr = document.createElement('tr');
      const fechaStr = u.fechaRegistro ? new Date(u.fechaRegistro).toLocaleString('es-VE', { hour12: false }) : '-';
      tr.innerHTML = `
        <td>${u.nombre || '-'}</td>
        <td>${u.telefono || '-'}</td>
        <td>${u.puntos ?? 0}</td>
        <td>${fechaStr}</td>
      `;
      if (Number(u.banned) === 1) tr.classList.add('banned-row');
      tbodyUsuarios.appendChild(tr);
    });
  }
  window.cargarUsuariosEnTabla = cargarUsuariosEnTabla;

  async function cargarPedidosPendientes() {
    if (!tbodyPagos) return;
    tbodyPagos.innerHTML = '';

    let pedidos = [];
    try {
      const res = await adminFetch(ADMIN_BE + 'admin.php?tipo=pedidos', { cache: 'no-store' });
      pedidos = await res.json();
    } catch (e) { pedidos = []; }

    if (!Array.isArray(pedidos) || pedidos.length === 0) {
      const tr = document.createElement('tr');
      const td = document.createElement('td');
      td.colSpan = 9;
      td.textContent = 'No hay pedidos pendientes.';
      tr.appendChild(td);
      tbodyPagos.appendChild(tr);
      return;
    }

    pedidos.forEach((p) => {
      const estado    = String(p.estado || 'pendiente').toLowerCase();
      const pendiente = (estado === 'pendiente');

      const montoUsd  = parseFloat(p.montoUsd) || 0;
      const tasa      = parseFloat(p.tasaBCV) || 1;
      const totalBs   = (montoUsd * tasa).toFixed(2);
      const f         = p.processedAt || p.fecha || p.createdAt || null;
      const fechaStr  = f ? new Date(f).toLocaleString('es-VE', { hour12:false }) : '-';

      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td><span class="orden-code" data-copy="${p.order_code || ''}">${p.order_code || '-'}</span></td>
        <td>${p.id || ''}</td>
        <td>${p.nombre || ''}</td>
        <td>${p.telefono || ''}</td>
       <td>
  ${pendiente
    ? `<input type="number" class="monto-usd" value="${montoUsd.toFixed(2)}" min="0" step="0.01" inputmode="decimal">`
    : `$${montoUsd.toFixed(2)}`
  }
</td>
<td class="col-$">$ ${totalBs}</td>

        <td>${fechaStr}</td>
        <td><span class="badge estado-${estado}">${estado}</span></td>
        <td class="acciones">
          ${pendiente
            ? `<button class="btn-aprobar-pedido" data-id="${p.id}" data-order="${p.order_code || ''}">Aprobar</button>
               <button class="btn-rechazar-pedido" data-id="${p.id}" data-order="${p.order_code || ''}">Rechazar</button>`
            : `<em>—</em>`
          }
        </td>
      `;
      tbodyPagos.appendChild(tr);
      if (pendiente) {
  const inp   = tr.querySelector('.monto-usd');
  const bsTd  = tr.querySelector('.col-$');
  inp.addEventListener('input', () => {
    const v = parseFloat(inp.value) || 0;
    bsTd.textContent = '$ ' + (v * tasa).toFixed(2);
  });
}

    });


    tbodyPagos.querySelectorAll('.btn-aprobar-pedido').forEach(btn => {
      btn.onclick = () => aprobarPedido({ id: btn.dataset.id, order_code: btn.dataset.order });
    });
    tbodyPagos.querySelectorAll('.btn-rechazar-pedido').forEach(btn => {
      btn.onclick = () => rechazarPedido(btn.dataset.id);
    });
  }
  window.cargarPedidosPendientes = cargarPedidosPendientes;

  async function aprobarPedido(rowOrId) {
    const payload = { accion: 'aprobar_pedido' };
    if (rowOrId && typeof rowOrId === 'object') {
      if (rowOrId.id) payload.id = Number(rowOrId.id);
      if (rowOrId.order_code) payload.order_code = String(rowOrId.order_code);
    } else {
      payload.id = Number(rowOrId);
    }
  // ...
 
  // NUEVO: si existe input editable en la fila, enviar montoUsd override
  try {
    const btn = document.querySelector(`.btn-aprobar-pedido[data-id="${payload.id}"]`);
    const row = btn && btn.closest('tr');
    const inp = row && row.querySelector('.monto-usd');
    if (inp) {
      const v = parseFloat(inp.value);
      if (isFinite(v) && v > 0) {
        payload.montoUsd = Number(v.toFixed(2));
      }
    }
  } catch (_) {}
  // ...

    let res, txt, data;
    try {
      res = await adminFetch(ADMIN_BE + 'admin.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });

      txt = await res.text();
      try { data = JSON.parse(txt); } catch {}
      if (!res.ok) throw new Error(`HTTP ${res.status}`);
      if (!data || !data.success) throw new Error((data && (data.error || data.message)) || (txt || 'Respuesta vacía'));

      const coins = (data.coinsAcredita ?? data.coinsSumados ?? '');
      showNeonAlert(`✅ Pedido aprobado${coins!=='' ? ' (+'+coins+' coins)' : ''}.`);
      await cargarPedidosPendientes();
    } catch (e) {
      console.error('[aprobarPedido] payload=', payload, 'resp=', txt);
      showNeonAlert('❌ No se pudo aprobar: ' + (e?.message || 'Error de red'));
    }
  }

  async function rechazarPedido(id) {
    try {
      const res = await adminFetch(ADMIN_BE + 'admin.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ accion: 'rechazar_pedido', id: Number(id) })
      });
      const txt = await res.text();
      let data = null; try { data = JSON.parse(txt); } catch {}
      if (data && data.success) {
        showNeonAlert('✅ Pedido rechazado.');
        await cargarPedidosPendientes();
      } else {
        showNeonAlert((data && (data.error || data.message)) || `No se pudo rechazar. Respuesta: ${txt.slice(0,180)}`);
      }
    } catch(e) {
      showNeonAlert('Error de red.');
    }
  }

  // === Solicitudes de canje ===
  function setHeadersSolicitudes() {
    const tbody = document.getElementById('tbody-solicitudes-canje');
    if (!tbody) return;
    const table = tbody.closest('table');
    const headRow = table?.querySelector('thead tr');
    if (!headRow) return;
    headRow.innerHTML = `
      <th>CÓDIGO</th>
      <th>NOMBRE</th>
      <th>TELÉFONO</th>
      <th>TIPO</th>
      <th>MONTO</th>
      <th>ESTADO</th>
      <th>FECHA</th>
      <th>ACCIONES</th>
    `;
  }

  async function renderizarSolicitudesCanje() {
    const tbody = document.getElementById('tbody-solicitudes-canje');
    if (!tbody) { console.warn('[Canjes] Falta #tbody-solicitudes-canje'); return; }

    setHeadersSolicitudes();
    tbody.innerHTML = `<tr><td colspan="8">Cargando…</td></tr>`;

    let lista = [];
    try {
      const res = await adminFetch(ADMIN_BE + 'admin.php?tipo=solicitudes', { cache: 'no-store' });
      const ct = res.headers.get('content-type') || '';
      if (!ct.includes('application/json')) {
        const txt = await res.text();
        console.error('[Canjes] Respuesta no-JSON:', txt);
        tbody.innerHTML = `<tr><td colspan="8">❌ Respuesta no válida</td></tr>`;
        showNeonAlert('❌ Respuesta no válida del servidor.');
        return;
      }
      lista = await res.json();
    } catch (e) {
      console.error('[Canjes] Error de red:', e);
      tbody.innerHTML = `<tr><td colspan="8">❌ Error de red</td></tr>`;
      showNeonAlert('❌ Error de red cargando solicitudes');
      return;
    }

    if (!Array.isArray(lista) || !lista.length) {
      tbody.innerHTML = `<tr><td colspan="8">Sin solicitudes</td></tr>`;
      return;
    }

    tbody.innerHTML = '';
    lista.forEach(sol => {
      const code   = sol.canje_code || sol.order_code || '-';
      const nombre = sol.nombre || '-';
      const tel    = sol.telefono || '-';
      const tipo   = sol.tipo || '-';
      const monto  = Number(sol.monto ?? 0);
      const estado = sol.estado || 'pendiente';
      const f      = sol.createdAt || sol.created_at || null;
      const fecha  = f ? new Date(f).toLocaleString('es-VE', { hour12:false }) : '-';
      const pendiente = String(estado).toLowerCase() === 'pendiente';

      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td><span class="orden-code" data-copy="${code}">${code}</span></td>
        <td>${nombre}</td>
        <td>${tel}</td>
        <td>${tipo}</td>
        <td>${monto}</td>
        <td>${estado}</td>
        <td>${fecha}</td>
        <td>
          ${pendiente ? `
            <button class="btn-aprobar-canje" data-id="${sol.id}">Aprobar</button>
            <button class="btn-rechazar-canje" data-id="${sol.id}">Rechazar</button>
          ` : ''}
        </td>
      `;
      tbody.appendChild(tr);
    });

    tbody.querySelectorAll('.btn-aprobar-canje').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.preventDefault();
        if (btn.dataset.busy === '1') return;
        btn.dataset.busy = '1'; btn.disabled = true;
        aprobarCanje(parseInt(btn.dataset.id, 10))
          .finally(() => { btn.dataset.busy = '0'; btn.disabled = false; });
      });
    });

    tbody.querySelectorAll('.btn-rechazar-canje').forEach(btn => {
      btn.addEventListener('click', (e) => {
        e.preventDefault();
        if (btn.dataset.busy === '1') return;
        btn.dataset.busy = '1'; btn.disabled = true;
        rechazarCanje(parseInt(btn.dataset.id, 10))
          .finally(() => { btn.dataset.busy = '0'; btn.disabled = false; });
      });
    });
  }

  async function aprobarCanje(id) {
    try {
      const res = await adminFetch(ADMIN_BE + 'admin.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ accion: 'aprobar_canje', id })
      });
      const data = await res.json();
      if (data.success) {
        showNeonAlert('✅ Solicitud de canje aprobada');
        renderizarSolicitudesCanje();
      } else {
        showNeonAlert(data.error || 'No se pudo aprobar.');
      }
    } catch (e) {
      console.error(e);
      showNeonAlert('Error de red: ' + e.message);
    }
  }

  async function rechazarCanje(id) {
    if (!Number.isInteger(id) || id <= 0) return;
    if (!confirm('¿Rechazar este canje? Esta acción no se puede deshacer.')) return;
    try {
      const res = await adminFetch(ADMIN_BE + 'admin.php', {
        method: 'POST',
        headers: {'Content-Type':'application/json'},
        body: JSON.stringify({ accion:'rechazar_canje', id, by:'admin' })
      });
      const j = await res.json();
      if (j && j.success) {
        showNeonAlert('✅ Canje rechazado');
        renderizarSolicitudesCanje();
      } else {
        showNeonAlert('❌ No se pudo rechazar: ' + (j?.error || 'Error desconocido'));
      }
    } catch (e) {
      console.error(e);
      showNeonAlert('❌ Error de red al rechazar');
    }
  }

  // =============================
  // Subir video (overlay progreso)
  // =============================
  (function(){
    function getPromoContainer(){
      return document.querySelector('#admin-panel .upload-box')
          || document.getElementById('admin-panel')
          || document.body;
    }
    function createPromoProgressUI() {
      let overlay = document.getElementById('promo-progress-wrap');
      if (!overlay) {
        overlay = document.createElement('div');
        overlay.id = 'promo-progress-wrap';
        overlay.className = 'promo-progress-overlay';
        overlay.innerHTML = `
          <div class="np-card" role="status" aria-live="polite">
            <div class="np-title">Subiendo video… <strong id="promo-progress-text">0%</strong></div>
            <div class="np-bar"><div id="promo-progress-bar"></div></div>
          </div>
        `;
        getPromoContainer().appendChild(overlay);
      }
      return overlay;
    }
    function updatePromoProgress(pct) {
      const bar = document.getElementById('promo-progress-bar');
      const txt = document.getElementById('promo-progress-text');
      if (bar) bar.style.width = pct + '%';
      if (txt) txt.textContent = pct + '%';
    }
    function removePromoProgressUI() {
      const overlay = document.getElementById('promo-progress-wrap');
      if (overlay && overlay.parentNode) overlay.parentNode.removeChild(overlay);
    }
    const promoInput = document.getElementById('promo-video-file');
    if (!promoInput) return;
    promoInput.addEventListener('change', () => {
      const file = promoInput.files && promoInput.files[0];
      if (!file) return;
      createPromoProgressUI();
      updatePromoProgress(0);
      const fd = new FormData();
      fd.append('video', file);
      const xhr = new XMLHttpRequest();
      xhr.open('POST', ADMIN_BE + 'upload_promo.php', true);
      xhr.upload.onprogress = (ev) => {
        if (ev.lengthComputable) {
          const pct = Math.max(0, Math.min(100, Math.round((ev.loaded / ev.total) * 100)));
          updatePromoProgress(pct);
        } else {
          updatePromoProgress((Date.now()/50)%100 | 0);
        }
      };
      xhr.onreadystatechange = () => {
        if (xhr.readyState !== 4) return;
        removePromoProgressUI();
        promoInput.value = '';
        let data = null; try { data = JSON.parse(xhr.responseText || '{}'); } catch {}
        if (xhr.status >= 200 && xhr.status < 300 && data && data.success) {
          showNeonAlert('✅ Video subido: ' + (data.src || 'OK'));
        } else {
          const msg = (data && (data.error || data.message)) || `Error HTTP ${xhr.status}`;
          showNeonAlert('❌ ' + msg);
        }
      };
      xhr.onerror = () => {
        removePromoProgressUI();
        promoInput.value = '';
        showNeonAlert('❌ Error de red subiendo el video.');
      };
      xhr.send(fd);
    });
  })();

  // =============================
  // Tasa BCV
  // =============================
  async function cargarTasaBCVAdmin() {
    try {
      const res = await fetch(BE + 'bcv.php');
      const data = await res.json();
      if (data.tasa) {
        const el = document.getElementById('tasa-bcv');
        if (el) el.textContent = `Tasa BCV: $ ${parseFloat(data.tasa).toFixed(2)}`;
      }
    } catch (e) {}
  }
  if (guardarTasaBtn){
    guardarTasaBtn.addEventListener('click', async () => {
      const nuevaTasa = parseFloat(String(manualTasaInput.value||'').trim().replace(',', '.'));
      if (isNaN(nuevaTasa) || nuevaTasa <= 0) return showNeonAlert('⚠️ Ingresa número válido mayor a 0.');
      try {
        const res = await adminFetch(ADMIN_BE + 'set_bcv.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ tasa: nuevaTasa })
        });
        const data = await res.json();
        if (data.success) {
          showNeonAlert(`💾 Tasa guardada: $ ${nuevaTasa.toFixed(2)}`);
          manualTasaInput.value = '';
          cargarTasaBCVAdmin();
        } else {
          showNeonAlert(data.error || 'No se pudo actualizar la tasa.');
        }
      } catch (e) {
        showNeonAlert('Error de red actualizando tasa.');
      }
    });
  }

  // =============================
  // MENSAJE POR CATEGORÍA (ADMIN + PÚBLICO)
  // =============================
// =============================
// MENSAJE POR CATEGORÍA (ADMIN + PÚBLICO) — Versión depurada
// =============================
let catSettingsCache = {};
let CAT_PUBLIC_URL = null;

// Solo referencia un contenedor existente; no crea HTML ni aplica estilos.
function getCategoryBannerEl() {
  return document.getElementById('category-message') || null;
}

// Fuente pública prioritaria: /api/categories.php (devuelve array)
// Fallback opcional: category-settings-public.php si algún día existe.
async function fetchCategorySettingsPublic() {
  // 1) /api/categories.php (o donde apunte window.BACKEND)
  try {
    const url1 = (window.BACKEND || '/api/').replace(/\/?$/, '/') + 'categories.php?_=' + Date.now();
    const r1 = await fetch(url1, { cache: 'no-store' });
    if (r1.ok) {
      const arr = await r1.json();
      if (Array.isArray(arr)) {
        const map = {};
        arr.forEach(row => {
          const raw = String(row.category || row.nombre || '').trim();
          const key = slugify(raw);
          const enabled = (row.enabled ?? row.is_active ?? row.active ?? 1);
          const msg = String(row.message ?? row.mensaje ?? '').trim();
          map[key] = { is_active: Number(enabled) ? 1 : 0, message: msg, raw };
        });
        catSettingsCache = map;
        return catSettingsCache;
      }
    }
  } catch (_) {}

  // 2) Fallback opcional si en el futuro agregas category-settings-public.php
  if (!CAT_PUBLIC_URL) {
    const bases = [
      (window.BACKEND || '/api/'),
      '/backend/',
      '/api/',
      '/'
    ];
    for (const base of bases) {
      const u = base.replace(/\/?$/, '/') + 'category-settings-public.php';
      try {
        const res = await fetch(u + '?_=' + Date.now(), { cache: 'no-store' });
        if (res.ok) { CAT_PUBLIC_URL = u; break; }
      } catch (_) {}
    }
  }
  if (CAT_PUBLIC_URL) {
    try {
      const r2 = await fetch(CAT_PUBLIC_URL + (CAT_PUBLIC_URL.includes('?') ? '&' : '?') + '_=' + Date.now(), { cache:'no-store' });
      if (r2.ok) {
        // Puede venir como mapa {cat:{...}} o array; normalizamos
        const data = await r2.json();
        const map = {};
        if (Array.isArray(data)) {
          data.forEach(row => {
            const raw = String(row.category || row.nombre || '').trim();
            const key = slugify(raw);
            const enabled = (row.enabled ?? row.is_active ?? row.active ?? 1);
            const msg = String(row.message ?? row.mensaje ?? '').trim();
            map[key] = { is_active: Number(enabled) ? 1 : 0, message: msg, raw };
          });
        } else if (data && typeof data === 'object') {
          Object.keys(data).forEach(k => {
            const raw = data[k].raw || k;
            const key = slugify(k);
            const enabled = (data[k].enabled ?? data[k].is_active ?? data[k].active ?? 1);
            const msg = String(data[k].message ?? data[k].mensaje ?? '').trim();
            map[key] = { is_active: Number(enabled) ? 1 : 0, message: msg, raw };
          });
        }
        catSettingsCache = map;
        return catSettingsCache;
      }
    } catch (_) {}
  }

  // Nada disponible
  catSettingsCache = {};
  return catSettingsCache;
}
window.fetchCategorySettingsPublic = fetchCategorySettingsPublic;

// Devuelve el texto para una categoría (acepta slug o nombre real)
window.getCategoryMessage = (cat) => {
  const key = slugify(cat);
  const rec = catSettingsCache[key];
  const msg = rec ? (rec.message || '') : '';
  return String(msg).trim();
};

// Lee la categoría activa del UI público
function getActiveCategoryKey() {
  const btn = document.querySelector('.categoria-btn.cat-active, .categoria-btn.active');
  return (btn && btn.dataset && btn.dataset.categoria) ? btn.dataset.categoria : 'all';
}

// Pinta/oculta el banner existente
async function applyCategoryBanner() {
  const banner = getCategoryBannerEl();
  if (!banner) return;

  if (!Object.keys(catSettingsCache).length) {
    await fetchCategorySettingsPublic(); // refresca cache
  }

  const key = getActiveCategoryKey();
  if (!key || key === 'all') { // en "Todos" no mostramos nada
    banner.textContent = '';
    banner.style.display = 'none';
    return;
  }

  const k = slugify(key);
  const rec = catSettingsCache[k] || null;
  const msg = rec ? String(rec.message || '').trim() : '';
  const act = rec ? (rec.is_active ?? rec.active ?? rec.activo ?? rec.enabled ?? 1) : 1;

  const visible = !!msg && String(act) !== '0';
  banner.textContent = msg;
  banner.style.display = visible ? 'block' : 'none';
}
window.applyCategoryBanner = applyCategoryBanner;

// -------- Admin UI (sin HTML inyectado) --------

// Si no hay labels en window, las pedimos al backend y construimos {slug: labelReal}
async function ensureCategoryLabels() {
  let labels = (window.categoryLabels || {});
  if (labels && Object.keys(labels).length) return labels;

  const base = (window.BACKEND_AUTH || '/backend/').replace(/\/?$/, '/');
  try {
    // 1) Obtenemos las categorías desde los productos ya cargados en la BD.
    const res = await fetch(base + 'products.php?admin=1', { cache: 'no-store' });
    const j = await res.json();
    const arr = Array.isArray(j) ? j : (Array.isArray(j?.data) ? j.data : []);
    const map = new Map();
    arr.forEach(p => {
      const raw = String(p.category || p.categoria || '').trim();
      if (!raw) return;
      const key = slugify(raw);
      if (!map.has(key)) map.set(key, raw);
    });

    // 2) Intentamos complementar con las categorías que el administrador ha
    // guardado explícitamente en la tabla categorias_estado.  Esto permite
    // mostrar en el selector categorías que no existan en la lista de
    // productos (por ejemplo, porque fueron cargadas manualmente o porque
    // actualmente no tienen productos asociados) para poder gestionar su
    // mensaje.
    try {
      const res2 = await fetch(base + 'category-settings.php', { cache: 'no-store', credentials: 'include' });
      const json2 = await res2.json();
      const data2 = (json2 && typeof json2 === 'object') ? (json2.data || json2) : null;
      if (data2 && typeof data2 === 'object') {
        Object.keys(data2).forEach(catKey => {
          const rec = data2[catKey];
          // catKey es la clave del objeto (nombre real) o usamos rec.category si existe
          const rawName = (rec && typeof rec === 'object' && rec.category) ? String(rec.category).trim() : String(catKey).trim();
          if (rawName) {
            const slug = slugify(rawName);
            if (!map.has(slug)) {
              map.set(slug, rawName);
            }
          }
        });
      }
    } catch (e) {
      // Silencioso: si falla la segunda petición, continuamos con lo que haya.
    }

    labels = {};
    for (const [k, v] of map) labels[k] = v;
    window.categoryLabels = labels;
  } catch (_) {
    labels = {};
  }
  return labels;
}

async function initCatMessageAdmin() {
  const box = document.getElementById('cat-msg-box');
  if (!box) return;

  const sel   = box.querySelector('#cat-msg-select');
  const txt   = box.querySelector('#cat-msg-text');
  const save  = box.querySelector('#cat-msg-save');
  const clear = box.querySelector('#cat-msg-clear');

  // 1) Poblar categorías con NOMBRE REAL como value (no slug)
  const labels = await ensureCategoryLabels();
  const cats = Object.keys(labels).sort((a, b) => (labels[a] || '').localeCompare(labels[b] || ''));
  sel.innerHTML =
    `<option value="" disabled selected>Selecciona una categoría</option>` +
    cats.map(k => {
      const labelReal = labels[k] || k;      // valor = nombre real, texto = nombre real
      return `<option value="${labelReal}">${labelReal}</option>`;
    }).join('');

  // 2) Precarga desde fuente pública normalizada
  let settings = await fetchCategorySettingsPublic();

  // Si hay una categoría activa en UI público, úsala
  const activeBtn = document.querySelector('.categoria-btn.cat-active, .categoria-btn.active');
  const activeCatRaw = activeBtn?.dataset?.categoria ? (labels[slugify(activeBtn.dataset.categoria)] || activeBtn.dataset.categoria) : '';
  if (activeCatRaw && [...sel.options].some(o => o.value === activeCatRaw)) {
    sel.value = activeCatRaw;
    txt.value = window.getCategoryMessage(activeCatRaw);
  }

  sel.addEventListener('change', () => {
    const raw = sel.value;
    txt.value = window.getCategoryMessage(raw);
  });

  async function postCatAdmin(bodyObj) {
    let resp, text, json = null;
    resp = await adminFetch((window.BACKEND_AUTH || '/backend/').replace(/\/?$/, '/') + 'category-settings.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(bodyObj)
    });
    text = await resp.text();
    try { json = JSON.parse(text); } catch {}
    if (!resp.ok) throw new Error(`HTTP ${resp.status}`);
    if (!json || json.success === false) throw new Error((json && (json.error || json.message)) || text || 'Respuesta inválida');
    return json;
  }

  save.addEventListener('click', async () => {
    const rawCategory = sel.value;
    if (!rawCategory) return showNeonAlert('⚠️ Selecciona una categoría.');
    const message = String(txt.value || '').trim();

    save.disabled = true; clear.disabled = true;
    try {
      // Guardamos con el NOMBRE REAL para que BD haga match exacto
      await postCatAdmin({ accion: 'set_message', category: rawCategory, message });

      // Releer público y refrescar cache normalizado (slug -> datos)
      settings = await fetchCategorySettingsPublic();

      // Forzamos el mensaje recién puesto en cache local también
      const k = slugify(rawCategory);
      settings[k] = settings[k] || { is_active: 1, message: '' };
      settings[k].message = message;
      catSettingsCache = settings;

      await applyCategoryBanner();
      showNeonAlert('💾 Mensaje guardado');
    } catch (e) {
      showNeonAlert('❌ No se pudo guardar: ' + (e?.message || 'Error'));
    } finally {
      save.disabled = false; clear.disabled = false;
    }
  });

  clear.addEventListener('click', async () => {
    const rawCategory = sel.value || '';
    if (!rawCategory) return showNeonAlert('⚠️ Selecciona una categoría.');
    clear.disabled = true; save.disabled = true;
    try {
      // Intento dedicado
      let ok = true;
      try {
        await postCatAdmin({ accion: 'clear_message', category: rawCategory });
      } catch (_) {
        ok = false;
      }
      // Fallback: set_message vacío
      if (!ok) {
        await postCatAdmin({ accion: 'set_message', category: rawCategory, message: '' });
      }

      settings = await fetchCategorySettingsPublic();
      txt.value = '';
      const k = slugify(rawCategory);
      if (settings[k]) settings[k].message = '';
      catSettingsCache = settings;

      await applyCategoryBanner();
      showNeonAlert('🧹 Mensaje limpiado');
    } catch (e) {
      showNeonAlert('❌ No se pudo limpiar: ' + (e?.message || 'Error'));
    } finally {
      clear.disabled = false; save.disabled = false;
    }
  });
}
window.initCatMessageAdmin = initCatMessageAdmin;

// (listener externo que ya tienes):
document.addEventListener('click', (e) => {
  const b = e.target.closest('.categoria-btn');
  if (b) setTimeout(applyCategoryBanner, 0);
});

  // Tabs del panel admin
  if (tabUsuarios && tabPagos && tabSolicitudes) {
    function activarTab(activa) {
      [tabUsuarios, tabPagos, tabSolicitudes].forEach(t => t.classList.remove('active'));
      activa.classList.add('active');
      contentUsuarios && (contentUsuarios.style.display = 'none');
      contentPagos && (contentPagos.style.display = 'none');
      contentSolicitudes && (contentSolicitudes.style.display = 'none');
    }
    tabUsuarios.addEventListener('click', () => {
      activarTab(tabUsuarios);
      contentUsuarios && (contentUsuarios.style.display = 'block');
      cargarUsuariosEnTabla();
    });
    tabPagos.addEventListener('click', () => {
      activarTab(tabPagos);
      contentPagos && (contentPagos.style.display = 'block');
      cargarPedidosPendientes();
    });
    tabSolicitudes.addEventListener('click', () => {
      activarTab(tabSolicitudes);
      contentSolicitudes && (contentSolicitudes.style.display = 'block');
      renderizarSolicitudesCanje();
    });
  }

  if (window.generateCategoryButtons) window.generateCategoryButtons();
  else {
    ensureCategoryBanner();
    fetchCategorySettingsPublic().then(applyCategoryBanner);
  }

  // Bandera "en vivo" por horario (12:00–18:00) para el banner de categoría
  (function(){
    const el = document.getElementById('category-message');
    if (!el) return;
    el.textContent = (el.textContent || '').trim();
    function tick(){
      const h = new Date().getHours();
      el.classList.toggle('is-live', h >= 12 && h < 18);
    }
    tick();
    setInterval(tick, 60_000);
  })();

  // =============================
  // DISPONIBILIDAD POR CATEGORÍA
  // =============================
  (function(){
    const BE2 = (window.BACKEND_AUTH || '/backend/').replace(/\/?$/, '/');
    const PRODUCTS_URL    = BE2 + 'products.php';
    const TOGGLE_PRIMARY  = BE2 + 'toggle-product.php';
    const TOGGLE_FALLBACK = [];

    const adminFetch = window.adminFetchGlobal || window.adminFetch || fetch;

    function normalizeProduct(p){
      const id  = Number(p.id ?? p.id_producto ?? p.product_id ?? 0);
      const nm  = String(p.name ?? p.nombre ?? '').trim();
      const cat = String(p.category ?? p.categoria ?? '').trim().toLowerCase();
      const av  = (p.available ?? p.disponible ?? p.is_active ?? 1);
      const available = (String(av)==='1' || av===true || av==='true');
      return { id, name:nm, category:cat, available };
    }

function ensureAvailabilityUI(){
  const host = document.querySelector('#admin-panel .upload-box');
  if (!host) return null;
  // Ya no creamos HTML ni <style>; todo vive en admin.html
  return document.getElementById('availability-box');
}


    let AV_ALL_PRODUCTS = [];
    let AV_FILTERED     = [];

    function fillCategoriesSelect(sel, productsForFallback){
      // Genera lista de categorías a partir de AV_ALL_PRODUCTS o productsForFallback
      const labels = (window.categoryLabels || {});
      let cats = [];
      if (Array.isArray(AV_ALL_PRODUCTS) && AV_ALL_PRODUCTS.length) {
        const set = new Set(AV_ALL_PRODUCTS.map(p => String(p.category || '').trim()).filter(Boolean));
        cats = [...set];
      } else if (Array.isArray(productsForFallback) && productsForFallback.length) {
        const set = new Set(productsForFallback.map(p => p.category).filter(Boolean));
        cats = [...set];
      } else {
        cats = Object.keys(labels);
      }
      // Añade 'all' como opción inicial
      cats = cats.sort((a,b) => {
        const la = labels[a] || a;
        const lb = labels[b] || b;
        return la.localeCompare(lb);
      });
      // Prepara HTML de opciones con etiqueta de 'Todas'
      const opts = [];
      opts.push('<option value="all">Todas</option>');
      cats.forEach(k => {
        const label = labels[k] || k;
        opts.push(`<option value="${k}">${label}</option>`);
      });
      sel.innerHTML = opts.join('');
    }

    async function loadProducts(cat){
const url = PRODUCTS_URL + (PRODUCTS_URL.includes('?') ? '&' : '?')
          + 'admin=1' + ((cat && cat !== 'all') ? '&cat=' + encodeURIComponent(cat) : '');

      try{
        const r = await fetch(url, {cache:'no-store'});
        const j = await r.json();
        const arr = Array.isArray(j) ? j : (Array.isArray(j?.data) ? j.data : []);
        const norm = arr.map(normalizeProduct).filter(x=>x.id && x.name);
        return cat && cat!=='all' ? norm.filter(p=>p.category===cat) : norm;
      }catch(e){ console.warn(e); return []; }
    }

    function renderList(list, host){
      host.innerHTML = '';
      if (!list || !list.length){
        host.innerHTML = `<div class="av-empty" style="opacity:.8;">No hay productos en esta categoría.</div>`;
        return;
      }
      const frag = document.createDocumentFragment();
      list.forEach(p => {
        const row = document.createElement('div');
        row.className = 'av-item';
        row.dataset.id = String(p.id);
        row.innerHTML = `
          <div class="av-name">${p.name}</div>
          <button class="av-delete-btn" type="button">Eliminar</button>
          <button class="av-state-btn ${p.available ? 'on' : 'off'}" type="button">
            ${p.available ? 'Disponible' : 'No disponible'}
          </button>
        `;
        const delBtn = row.querySelector('.av-delete-btn');
        const stateBtn = row.querySelector('.av-state-btn');
        // Maneja el borrado del producto
        delBtn.addEventListener('click', async () => {
          if (!p.id) return;
          if (!confirm('¿Eliminar producto? Esta acción no se puede deshacer.')) return;
          try {
            const res = await adminFetch(ADMIN_BE + 'delete-product.php', {
              method: 'POST',
              headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
              body: new URLSearchParams({ id: p.id })
            });
            const data = await res.json();
            if (!data || data.success === false) {
              showNeonAlert(data && data.error ? data.error : 'Error al eliminar producto');
            } else {
              showNeonAlert('🗑️ Producto eliminado');
              // Elimina de los arreglos en memoria
              AV_ALL_PRODUCTS = AV_ALL_PRODUCTS.filter(x => x.id !== p.id);
              AV_FILTERED = AV_FILTERED.filter(x => x.id !== p.id);
              // Recarga la lista
              applyFilter();
            }
          } catch (err) {
            console.error(err);
            showNeonAlert('Error al eliminar producto');
          }
        });
        // Maneja el cambio de disponibilidad
        stateBtn.addEventListener('click', async () => {
          const newState = !p.available;
          stateBtn.disabled = true;
          const ok = await toggleOne(p.id, newState);
          stateBtn.disabled = false;
          if (ok) {
            p.available = newState;
            stateBtn.classList.toggle('on', newState);
            stateBtn.classList.toggle('off', !newState);
            stateBtn.textContent = newState ? 'Disponible' : 'No disponible';
            document.dispatchEvent(new CustomEvent('admin:availability-changed', {
              detail: { ids: [p.id], available: newState }
            }));
          }
        });
        frag.appendChild(row);
      });
      host.appendChild(frag);
    }

    function applyFilter(){
      const selEl = document.getElementById('av-cat');
      const q = (document.getElementById('av-search').value || '').toLowerCase().trim();
      const selectedCat = (selEl && selEl.value) ? selEl.value : 'all';
      AV_FILTERED = AV_ALL_PRODUCTS.filter(p => {
        // Filtra por categoría si no es 'all'
        const matchCat = selectedCat === 'all' || String(p.category || '') === String(selectedCat);
        // Filtra por nombre si hay búsqueda
        const matchSearch = !q || (p.name && p.name.toLowerCase().includes(q));
        return matchCat && matchSearch;
      });
      renderList(AV_FILTERED, document.getElementById('av-list'));
    }

    async function toggleOne(id, available){
      const bodyJSON = JSON.stringify({ id, available: available?1:0, is_active: available?1:0 });
      try{
        const r = await adminFetch(TOGGLE_PRIMARY, {
          method:'POST', headers:{'Content-Type':'application/json'}, body: bodyJSON
        });
        if (r.ok){
          let j = null; try{ j = await r.clone().json(); }catch(_){}
          if (!j || j.success !== false) {
            window.showNeonAlert && window.showNeonAlert(available ? '✅ Marcado disponible' : '✅ Marcado no disponible');
            return true;
          }
        }
      }catch(_){}

      for (const url of TOGGLE_FALLBACK){
        try{
          const r = await adminFetch(url, { method:'POST', headers:{'Content-Type':'application/json'}, body: bodyJSON });
          if (r.ok) return true;
        }catch(_){}
      }

      const form = new URLSearchParams({ id:String(id), available:available?'1':'0', is_active:available?'1':'0' });
      try{
        const r = await adminFetch(TOGGLE_PRIMARY, {
          method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body: form.toString()
        });
        if (r.ok) return true;
      }catch(_){}

      window.showNeonAlert && window.showNeonAlert('❌ No se pudo cambiar el estado');
      return false;
    }

    async function initAvailabilityAdmin(){
      if (window.__availInitDone) return;
      const sec = ensureAvailabilityUI(); if (!sec) return;
      window.__availInitDone = true;

      const sel     = sec.querySelector('#av-cat');
      const search  = sec.querySelector('#av-search');
      const listBox = sec.querySelector('#av-list');

      // Carga inicial: obtiene todos los productos (todas las categorías)
      AV_ALL_PRODUCTS = await loadProducts('all');
      // Llenar select con categorías disponibles
      fillCategoriesSelect(sel, AV_ALL_PRODUCTS);

      // Selecciona categoría activa (si existe) o 'all'
      const activeBtn = document.querySelector('.categoria-btn.cat-active, .categoria-btn.active');
      const activeCat = (activeBtn?.dataset?.categoria) || 'all';
      if ([...sel.options].some(o => o.value === activeCat)) sel.value = activeCat;
      else sel.value = 'all';

      function refresh(){
        // Filtra productos en memoria según la categoría seleccionada y el texto de búsqueda
        applyFilter();
      }

      // Al cambiar la categoría o escribir en búsqueda, filtra
      sel.addEventListener('change', refresh);
      search.addEventListener('input', refresh);

      // Llama a applyFilter inicialmente para pintar la lista
      applyFilter();
    }
    window.initAvailabilityAdmin = initAvailabilityAdmin;
  })();

  // =============================
  // Finanzas Admin
  // =============================
  function initFinanzasAdmin(){
    const desdeEl = document.getElementById('finanzas-desde');
    const hastaEl = document.getElementById('finanzas-hasta');
    const btn     = document.getElementById('finanzas-cargar-btn');
    const kpisBox = document.getElementById('finanzas-kpis');
    const tbody   = document.getElementById('finanzas-tbody');
    if (!desdeEl || !hastaEl || !btn || !kpisBox || !tbody) return;

    const hoy = new Date().toISOString().split('T')[0];
    if (!desdeEl.value) desdeEl.value = hoy;
    if (!hastaEl.value) hastaEl.value = hoy;

    async function cargar(){
      const d = desdeEl.value || hoy;
      const h = hastaEl.value || d;
      try {
        const res = await adminFetch(ADMIN_BE + 'finanzas.php?desde=' + encodeURIComponent(d) + '&hasta=' + encodeURIComponent(h));
        const data = await res.json();
        if (!data || data.success === false) {
          showNeonAlert(data && data.error ? data.error : 'Error al cargar finanzas');
          return;
        }
        renderKpis(data.summary);
        renderFinanzasTable(data.records);
      } catch (e) {
        console.error(e);
        showNeonAlert('Error al cargar finanzas');
      }
    }

    function renderKpis(summary){
      kpisBox.innerHTML = '';
      if (!summary) return;
      const rec = summary.pedidos || { total:0, count:0 };
      const sol = summary.solicitudes || { total:0, count:0 };
      const cards = [
        { title:'Total Ventas ', value: (rec.total || 0).toFixed(2) },
        { title:'Total Canjes ', value: (sol.total || 0).toFixed(2) },
        { title:'# N° Recargas', value: rec.count || 0 },
        { title:'# N° Canjes', value: sol.count || 0 }
      ];
      const frag = document.createDocumentFragment();
      cards.forEach(c => {
        const div = document.createElement('div');
        div.className = 'finanzas-kpi-card';
        const h5 = document.createElement('h5');
        h5.textContent = c.title;
        const span = document.createElement('div');
        span.className = 'kpi-value';
        span.textContent = c.value;
        div.appendChild(h5);
        div.appendChild(span);
        frag.appendChild(div);
      });
      kpisBox.appendChild(frag);
    }

    function renderFinanzasTable(rows){
      tbody.innerHTML = '';
      if (!rows || !rows.length) return;
      const frag = document.createDocumentFragment();
      rows.forEach(r => {
        const tr = document.createElement('tr');
        const tdDate = document.createElement('td');
        tdDate.textContent = r.date || '';
        const rec = r.pedidos || { total:0, count:0 };
        const sol = r.solicitudes || { total:0, count:0 };
        const tdRecUsd = document.createElement('td');
        tdRecUsd.textContent = (rec.total || 0).toFixed(2);
        const tdCanj = document.createElement('td');
        tdCanj.textContent = (sol.total || 0).toFixed(2);
        const tdRecCnt = document.createElement('td');
        tdRecCnt.textContent = rec.count || 0;
        const tdCanjCnt = document.createElement('td');
        tdCanjCnt.textContent = sol.count || 0;
        tr.appendChild(tdDate);
        tr.appendChild(tdRecUsd);
        tr.appendChild(tdCanj);
        tr.appendChild(tdRecCnt);
        tr.appendChild(tdCanjCnt);
        frag.appendChild(tr);
      });
      tbody.appendChild(frag);
    }

    btn.addEventListener('click', cargar);
    cargar();

    const exportCsvBtn = document.getElementById('finanzas-export-csv');
    const exportXlsBtn = document.getElementById('finanzas-export-excel');

    async function exportFinanzas(format){
      const d = (desdeEl.value || hoy);
      const h = (hastaEl.value || d);
      const url = `${BE}finanzas.php?desde=${encodeURIComponent(d)}&hasta=${encodeURIComponent(h)}&export=${format}`;
      try {
        const res = await adminFetch(url);
        if (!res.ok) {
          showNeonAlert('Error al exportar');
          return;
        }
        const blob = await res.blob();
        let filename = '';
        const disp = res.headers.get('content-disposition');
        if (disp) {
          const match = disp.match(/filename="?([^";]+)"?/i);
          if (match && match[1]) filename = match[1];
        }
        if (!filename) {
          const ext = (format === 'excel' ? 'xls' : 'csv');
          filename = `finanzas_${d}_al_${h}.${ext}`;
        }
        const urlBlob = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = urlBlob;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(urlBlob);
      } catch (err) {
        console.error(err);
        showNeonAlert('Error al exportar');
      }
    }
    if (exportCsvBtn) exportCsvBtn.onclick = () => exportFinanzas('csv');
    if (exportXlsBtn) exportXlsBtn.onclick = () => exportFinanzas('excel');
  }

  // =============================
  // Agregar Producto Admin
  // =============================
  function initAddProductAdmin(){
    const btn = document.getElementById('prod-add-btn');
    const fileInput = document.getElementById('prod-image');
    const fileNameSpan = document.getElementById('prod-image-name');
    if (fileInput) {
      fileInput.addEventListener('change', () => {
        const file = fileInput.files && fileInput.files[0];
        if (fileNameSpan) fileNameSpan.textContent = file ? file.name : '';
      });
    }
    loadProducts();
    if (!btn) return;
    btn.addEventListener('click', async () => {
      const cat  = (document.getElementById('prod-category').value || '').trim() || 'otros';
      const name = (document.getElementById('prod-name').value || '').trim();
      const desc = (document.getElementById('prod-desc').value || '').trim();
      const priceVal = document.getElementById('prod-price').value;
      const price = parseFloat(priceVal);
      if (!name || !price || price <= 0) {
        showNeonAlert('⚠️ Nombre y precio válidos son obligatorios');
        return;
      }
      const formData = new FormData();
      formData.append('category', cat);
      formData.append('name', name);
      formData.append('desc', desc);
      formData.append('price', price);
      if (fileInput && fileInput.files && fileInput.files[0]) {
        formData.append('image', fileInput.files[0]);
      }
      try {
  const res = await adminFetch(ADMIN_BE + 'add-product.php', { method:'POST', body: formData });
  const ct = (res.headers.get('content-type') || '').toLowerCase();
  let data = null;
  if (ct.includes('application/json')) {
    data = await res.json();
  } else {
    const txt = await res.text();
    throw new Error(txt.slice(0, 300) || `HTTP ${res.status}`);
  }
  if (!data || data.success === false) {
    throw new Error((data && data.error) || 'Error al agregar producto');
  }
  showNeonAlert('✅ Producto añadido');
  // ... limpieza y recarga ...
} catch (e) {
  console.error(e);
  showNeonAlert('❌ ' + (e.message || e));
}

    });
  }

  // =============================
  // Productos: cargar y eliminar
  // =============================
  async function loadProducts(){
    const container = document.getElementById('prod-list');
    if (!container) return;
    try {
      const res = await fetch(BE + 'products.php');
      const data = await res.json();
      container.innerHTML = '';
      if (!Array.isArray(data) || !data.length) {
        container.textContent = 'No hay productos.';
        return;
      }
      const frag = document.createDocumentFragment();
      data.forEach(p => {
        const item = document.createElement('div');
        item.className = 'product-item';
        const info = document.createElement('div');
        info.className = 'prod-info';
        const nameEl = document.createElement('div');
        nameEl.textContent = `${p.name} (${p.category})`;
        const priceEl = document.createElement('div');
        priceEl.textContent = `$${parseFloat(p.price).toFixed(2)}`;
        info.appendChild(nameEl);
        info.appendChild(priceEl);
        const actions = document.createElement('div');
        actions.className = 'prod-actions';
        const delBtn = document.createElement('button');
        delBtn.textContent = 'Eliminar';
        delBtn.addEventListener('click', () => deleteProduct(p.id));
        actions.appendChild(delBtn);
        item.appendChild(info);
        item.appendChild(actions);
        frag.appendChild(item);
      });
      container.appendChild(frag);
    } catch (err) {
      console.error(err);
      container.textContent = 'Error al cargar productos';
    }
  }

  // =============================
  // Modal de Finanzas: abrir y cerrar
  // =============================
  (function(){
    const openBtn = document.getElementById('open-finanzas-modal');
    const modal   = document.getElementById('finanzas-modal');
    const closeBtn= document.getElementById('finanzas-close');
    if (openBtn && modal && closeBtn) {
      openBtn.addEventListener('click', () => {
        modal.style.display = 'flex';
        initFinanzasAdmin();
      });
      closeBtn.addEventListener('click', () => {
        modal.style.display = 'none';
      });
      modal.addEventListener('click', (e) => {
        if (e.target === modal) {
          modal.style.display = 'none';
        }
      });
    }
  })();

  // Auto-inicializa al abrir el panel
  const panel = document.getElementById('admin-panel');
  if (panel){
    const tryInit = ()=>{
      const visible = panel.offsetParent !== null || getComputedStyle(panel).display !== 'none';
      if (visible) initAvailabilityAdmin();
        initCatMessageAdmin?.();
    };
    tryInit();
    const mo = new MutationObserver(tryInit);
    mo.observe(panel, { attributes:true, attributeFilter:['style','class'] });
    document.addEventListener('admin:ready', initAvailabilityAdmin);
  }

});
