<?php
// upload-products.php
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

// Protecciones de acceso
require_once __DIR__ . '/guard.php';
require_admin();
require_csrf_for_write();

require_once __DIR__ . '/db.php';
$mysqli->set_charset('utf8mb4');
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); // errores como excepciones

// === utilidades ===
function slugify($s) {
    $s = trim((string)($s ?? ''));
    if ($s === '') return 'otros';
    // quitar acentos
    if (function_exists('transliterator_transliterate')) {
        $s = transliterator_transliterate('Any-Latin; Latin-ASCII', $s);
    } else {
        $tmp = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
        if ($tmp !== false) $s = $tmp;
    }
    $s = strtolower($s);
    $s = preg_replace('/[^a-z0-9]+/', '-', $s);
    $s = preg_replace('/^-+|-+$/', '', $s);
    return $s ?: 'otros';
}

function force_webp_path($category, $imgIn) {
    // Construye /imgproductos/<slug>/<basename>.webp
    $slug = slugify($category ?: 'otros');
    $path = (string)($imgIn ?? '');
    $path = trim($path);

    if ($path === '') return ''; // sin imagen

    // Si es URL absoluta, extraer solo la parte de path
    if (filter_var($path, FILTER_VALIDATE_URL)) {
        $parsedPath = parse_url($path, PHP_URL_PATH);
        $base = basename($parsedPath ?: '');
    } else {
        // Si es ruta/relativa/absoluta, quedarnos con el basename
        $base = basename(str_replace('\\', '/', $path));
    }

    if ($base === '' || $base === '.' || $base === '..') return '';

    // Quitar extensión original y forzar .webp
    $baseSinExt = preg_replace('/\.[^.]+$/', '', $base);
    return "/imgproductos/{$slug}/{$baseSinExt}.webp";
}

// === entrada ===
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);

if (!$in || !isset($in['products']) || !is_array($in['products'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'No se recibieron productos válidos']); exit;
}

// Si quieres reemplazo total en cada carga, deja true. Si no, pon false.
$REPLACE_ALL = true;

try {
    $mysqli->begin_transaction();

    if ($REPLACE_ALL) {
        $mysqli->query("TRUNCATE TABLE `productos`");
    }

    $stmt = $mysqli->prepare("INSERT INTO `productos` (`category`,`name`,`desc`,`price`,`img`) VALUES (?,?,?,?,?)");

    $insertados = 0;
    $omitidos   = 0;

    foreach ($in['products'] as $p) {
        $cat   = $p['category']     ?? $p['categoria']   ?? 'otros';
        $name  = $p['name']         ?? $p['nombre']      ?? '';
        $desc  = $p['desc']         ?? $p['descripcion'] ?? '';
        // price puede venir con coma decimal en el admin, pero ya lo parseas allí; por si acaso:
        $price = $p['price'] ?? $p['precio'] ?? 0;
        if (!is_numeric($price)) {
            $price = floatval(str_replace(',', '.', (string)$price));
        } else {
            $price = floatval($price);
        }
        $imgIn = $p['img']          ?? $p['imagen']      ?? '';

        // Normalizar ruta de imagen al formato requerido
        $imgOut = force_webp_path($cat, $imgIn);

        // Reglas mínimas: nombre y precio > 0
        if (!$name || $price <= 0) {
            $omitidos++;
            continue;
        }

        $stmt->bind_param("sssds", $cat, $name, $desc, $price, $imgOut);
        $stmt->execute();
        $insertados++;
    }

    $stmt->close();
    $mysqli->commit();

    echo json_encode([
        'success'    => true,
        'insertados' => $insertados,
        'omitidos'   => $omitidos
    ]);

} catch (Throwable $e) {
    if ($mysqli->errno) {
        // noop
    }
    $mysqli->rollback();
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
