<?php
// /backend/add-product.php — Acepta cualquier imagen y convierte a .webp (robusto y con diagnóstico)
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', '0');

require_once __DIR__ . '/db.php';
$mysqli->set_charset('utf8mb4');

function out($arr, int $code=200){ http_response_code($code); echo json_encode($arr, JSON_UNESCAPED_UNICODE); exit; }
function log_err($msg){ error_log('[add-product] '.$msg); }

function slugify($s){
  $s = trim((string)$s);
  if ($s==='') return 'otros';
  if (function_exists('transliterator_transliterate')) $s = transliterator_transliterate('Any-Latin; Latin-ASCII', $s);
  else { $tmp=@iconv('UTF-8','ASCII//TRANSLIT//IGNORE',$s); if($tmp!==false) $s=$tmp; }
  $s = strtolower($s);
  $s = preg_replace('/[^a-z0-9]+/','-',$s);
  $s = preg_replace('/^-+|-+$/','',$s);
  return $s ?: 'otros';
}
function safe_name($s){
  $s = trim($s);
  $s = preg_replace('/\.[^.]+$/','',$s);
  $s = preg_replace('/[^\p{L}\p{N}\s\-\&]/u','', $s);
  $s = preg_replace('/\s+/',' ', $s);
  return $s !== '' ? $s : ('img '.date('Ymd_His'));
}
function detect_mime(string $path): string {
  if (function_exists('finfo_open')) {
    $fi = finfo_open(FILEINFO_MIME_TYPE);
    if ($fi) { $m = finfo_file($fi, $path); finfo_close($fi); if ($m) return strtolower($m); }
  }
  if (function_exists('mime_content_type')) {
    $m = @mime_content_type($path);
    if ($m) return strtolower($m);
  }
  $gi = @getimagesize($path);
  return is_array($gi) && isset($gi['mime']) ? strtolower($gi['mime']) : '';
}

// ---- Conversión a WEBP segura (GD si existe; si no, Imagick; si no, error controlado) ----
function convert_to_webp(string $srcPath, string $dstPath, int $quality=82): bool {
  $mime = detect_mime($srcPath);
  $ext  = strtolower(pathinfo($srcPath, PATHINFO_EXTENSION));

  // 1) Si GD está disponible (y imagewebp existe)
  if (extension_loaded('gd') && function_exists('imagewebp')) {
    $img = null;
    try {
      if (($mime==='image/jpeg' || $ext==='jpg' || $ext==='jpeg') && function_exists('imagecreatefromjpeg')) {
        $img = @imagecreatefromjpeg($srcPath);
        if ($img && function_exists('exif_read_data')) {
          $exif = @exif_read_data($srcPath);
          $o = $exif['Orientation'] ?? 1;
          if (in_array($o, [3,6,8], true)) {
            if ($o==3) $img = imagerotate($img, 180, 0);
            if ($o==6) $img = imagerotate($img, -90, 0);
            if ($o==8) $img = imagerotate($img, 90, 0);
          }
        }
      } elseif (($mime==='image/png' || $ext==='png') && function_exists('imagecreatefrompng')) {
        $img = @imagecreatefrompng($srcPath);
        if ($img) {
          if (function_exists('imagepalettetotruecolor')) @imagepalettetotruecolor($img);
          @imagealphablending($img, true);
          @imagesavealpha($img, true);
        }
      } elseif (($mime==='image/gif' || $ext==='gif') && function_exists('imagecreatefromgif')) {
        $img = @imagecreatefromgif($srcPath);
        if ($img) {
          if (function_exists('imagepalettetotruecolor')) @imagepalettetotruecolor($img);
          @imagealphablending($img, true);
          @imagesavealpha($img, true);
        }
      } elseif (($mime==='image/bmp' || $ext==='bmp') && function_exists('imagecreatefrombmp')) {
        $img = @imagecreatefrombmp($srcPath);
      } elseif (($mime==='image/webp' || $ext==='webp') && function_exists('imagecreatefromwebp')) {
        $img = @imagecreatefromwebp($srcPath);
      } elseif (function_exists('imagecreatefromstring')) {
        $data = @file_get_contents($srcPath);
        if ($data!==false) $img = @imagecreatefromstring($data);
      }
      if ($img) {
        $ok = @imagewebp($img, $dstPath, $quality);
        @imagedestroy($img);
        if ($ok) return true;
      }
    } catch (Throwable $e) { log_err('GD failed: '.$e->getMessage()); }
  }

  // 2) Fallback: Imagick
  if (extension_loaded('imagick')) {
    try {
      $im = new Imagick();
      $im->readImage($srcPath);
      if (method_exists($im,'getImageOrientation') && method_exists($im,'setImageOrientation')) {
        if (defined('Imagick::ORIENTATION_TOPLEFT') && $im->getImageOrientation() !== Imagick::ORIENTATION_TOPLEFT) {
          $im->setImageOrientation(Imagick::ORIENTATION_TOPLEFT);
        }
      }
      $im->setImageFormat('webp');
      if (method_exists($im,'setImageAlphaChannel')) $im->setImageAlphaChannel(Imagick::ALPHACHANNEL_ACTIVATE);
      if (method_exists($im,'setImageCompressionQuality')) $im->setImageCompressionQuality($quality);
      $ok = $im->writeImage($dstPath);
      $im->clear(); $im->destroy();
      if ($ok) return true;
    } catch (Throwable $e) { log_err('Imagick failed: '.$e->getMessage()); }
  }

  return false;
}

// ---- Diagnóstico rápido: GET /backend/add-product.php?diag=1
if (isset($_GET['diag'])) {
  $docroot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? dirname(__DIR__), '/');
  $testDir = $docroot.'/imgproductos/__diag';
  @mkdir($testDir, 0755, true);
  $writable = is_writable($docroot.'/imgproductos') || is_writable($testDir);
  out([
    'gd_loaded'      => extension_loaded('gd'),
    'imagewebp'      => function_exists('imagewebp'),
    'jpeg'           => function_exists('imagecreatefromjpeg'),
    'png'            => function_exists('imagecreatefrompng'),
    'gif'            => function_exists('imagecreatefromgif'),
    'bmp'            => function_exists('imagecreatefrombmp'),
    'imagick_loaded' => extension_loaded('imagick'),
    'exif_available' => function_exists('exif_read_data'),
    'docroot'        => $docroot,
    'img_dir'        => $docroot.'/imgproductos',
    'writable'       => $writable
  ]);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') out(['success'=>false,'error'=>'Método no permitido'],405);

// Campos
$category = $_POST['category'] ?? $_POST['categoria'] ?? 'otros';
$name     = $_POST['name']     ?? $_POST['nombre']    ?? '';
$desc     = $_POST['desc']     ?? $_POST['descripcion']?? '';
$priceRaw = $_POST['price']    ?? $_POST['precio']    ?? '';
$price    = is_numeric($priceRaw) ? (float)$priceRaw : (float)str_replace(',','.',$priceRaw);
if ($name==='' || $price<=0) out(['success'=>false,'error'=>'Nombre y precio válidos son obligatorios'],400);

// Imagen (opcional)
$imgRel = '';
if (isset($_FILES['image']) && $_FILES['image']['error'] !== UPLOAD_ERR_NO_FILE) {
  if ($_FILES['image']['error'] !== UPLOAD_ERR_OK) out(['success'=>false,'error'=>'Error de subida ('.$_FILES['image']['error'].')'],400);

  $tmp  = $_FILES['image']['tmp_name'];
  $orig = $_FILES['image']['name'];

  // Validación básica de imagen
  $gi = @getimagesize($tmp);
  if ($gi === false) out(['success'=>false,'error'=>'Archivo no es una imagen válida'],400);

  $catSlug  = slugify($category);
  $baseName = safe_name($orig) . '.webp';

  $docroot  = rtrim($_SERVER['DOCUMENT_ROOT'] ?? dirname(__DIR__), '/');
  $targetDir= $docroot . '/imgproductos/' . $catSlug;
  if (!is_dir($targetDir) && !@mkdir($targetDir, 0755, true)) {
    log_err('mkdir fallo: '.$targetDir);
    out(['success'=>false,'error'=>'No se pudo crear carpeta destino'],500);
  }
  if (!is_writable($targetDir)) {
    log_err('no writable: '.$targetDir);
    out(['success'=>false,'error'=>'Carpeta destino sin permisos de escritura'],500);
  }

  // Evita colisión
  $target = $targetDir . '/' . $baseName;
  if (file_exists($target)) {
    $baseName = safe_name($orig) . ' ' . date('Ymd_His') . '.webp';
    $target   = $targetDir . '/' . $baseName;
  }

  // Convertir a webp
  if (!convert_to_webp($tmp, $target, 82)) {
    @unlink($tmp);
    $caps = @file_get_contents('php://input'); // no sensible; solo por si acaso
    log_err('conversion fallo (GD/Imagick ausentes o error)');
    out(['success'=>false,'error'=>'No se pudo convertir la imagen a .webp (GD/Imagick no disponible o fallo de conversión)'],500);
  }
  @chmod($target, 0644);

  $imgRel = '/imgproductos/'.$catSlug.'/'.$baseName;
}

// Insertar en BD
$stmt = $mysqli->prepare("INSERT INTO productos (`category`,`name`,`desc`,`price`,`img`,`is_active`) VALUES (?,?,?,?,?,1)");
if (!$stmt) out(['success'=>false,'error'=>'DB prepare: '.$mysqli->error],500);
$stmt->bind_param('sssds', $category, $name, $desc, $price, $imgRel);
if (!$stmt->execute()) out(['success'=>false,'error'=>'DB: '.$stmt->error],500);

out(['success'=>true,'id'=>$stmt->insert_id,'img'=>$imgRel]);
