<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
ini_set('display_errors','0');

require_once dirname(__DIR__) . '/backend/db.php';
$mysqli->set_charset('utf8mb4');

// ================== IMG RESOLVER (tolerante) ==================
// Busca la imagen ignorando maysculas, acentos, espacios, guiones y extensin.
// Soporta rutas relativas y absolutas (http/https). Prefiere .webp si existe.
function resolve_img_ci(string $rel): string {
  $rel = trim($rel ?? '');
  $PLACE = '/imgproductos/placeholder.png';

  // Absoluta (CDN) => devulvela
  if ($rel !== '' && preg_match('~^https?://~i', $rel)) return $rel;

  // Asegura slash inicial y separa query (?v=...)
  if ($rel === '') $rel = $PLACE;
  if ($rel[0] !== '/') $rel = '/' . ltrim($rel, '/');
  $query = '';
  if (($qpos = strpos($rel, '?')) !== false) {
    $query = substr($rel, $qpos);
    $rel   = substr($rel, 0, $qpos);
  }

  $docroot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? dirname(__DIR__), '/');

  // 0) Coincidencia exacta
  if (is_file($docroot . $rel)) return $rel . $query;

  // Normalizador (quita acentos/09 y espacios/guiones/_ y baja a minsculas)
  $norm = function (string $s): string {
    $s = trim($s);
    if ($s === '') return '';
    $s = str_replace('\\', '/', $s);
    $s = preg_replace('~/+~', '/', $s);
    $s = iconv('UTF-8', 'ASCII//TRANSLIT', $s);
    $s = strtolower($s);
    $s = preg_replace('~[\s\-_]+~', '', $s);
    return $s;
  };

  // Recorre cada segmento de la ruta intentando emparejar por nombre "normalizado"
  $segments = array_values(array_filter(explode('/', ltrim($rel, '/'))));
  if (!$segments) return $PLACE;

  $absPath = $docroot;
  $rebuilt = '';
  $lastIdx = count($segments) - 1;

  // 1) Resolver carpetas (todos los segmentos menos el ltimo)
  for ($i = 0; $i < $lastIdx; $i++) {
    $seg = $segments[$i];
    if (!is_dir($absPath)) return $PLACE;

    $entries = @scandir($absPath);
    if (!$entries) return $PLACE;

    $match = null;
    $segN = $norm($seg);
    foreach ($entries as $e) {
      if ($e === '.' || $e === '..') continue;
      if (!is_dir($absPath . '/' . $e)) continue;
      if ($norm($e) === $segN) { $match = $e; break; }
    }
    if ($match === null) return $PLACE;

    $absPath .= '/' . $match;
    $rebuilt .= '/' . $match;
  }

  // 2) Resolver archivo (ltimo segmento): probar extensiones y coincidencia de nombre
  $fileSeg = $segments[$lastIdx];
  if (!is_dir($absPath)) return $PLACE;

  // base name sin extensin del ltimo segmento (normalizado)
  $base = pathinfo($fileSeg, PATHINFO_FILENAME);
  $baseN = $norm($base);

  // Extensiones candidatas (preferencia por .webp)
  $exts = ['webp','png','jpg','jpeg','gif','JPG','JPEG','PNG','WEBP'];

  // a) si el nombre exacto con alguna extensin existe
  foreach ($exts as $ext) {
    $try = $rebuilt . '/' . $base . '.' . $ext;
    if (is_file($docroot . $try)) return $try . $query;
  }

  // b) buscar por coincidencia normalizada dentro de la carpeta
  $entries = @scandir($absPath);
  if ($entries) {
    foreach ($exts as $ext) {
      foreach ($entries as $e) {
        if ($e === '.' || $e === '..') continue;
        $p = $absPath . '/' . $e;
        if (!is_file($p)) continue;
        $extE = pathinfo($e, PATHINFO_EXTENSION);
        if (strtolower($extE) !== strtolower($ext)) continue;
        $nameE = pathinfo($e, PATHINFO_FILENAME);
        if ($norm($nameE) === $baseN) {
          return $rebuilt . '/' . $e . $query;
        }
      }
    }
  }

  // c) nada encontrado  placeholder
  return $PLACE;
}

// ================== PARAMS ==================
$cat   = isset($_GET['cat'])   ? trim((string)$_GET['cat']) : '';
$q     = isset($_GET['q'])     ? trim((string)$_GET['q'])   : '';
$all   = isset($_GET['all'])   ? ((int)$_GET['all']===1)    : false;
$debug = isset($_GET['debug']) ? ((int)$_GET['debug']===1)  : false;

// ================== WHERE ==================
$where = [];
$params = [];
$types  = '';

if (!$all) {
  $where[] = "is_active = 1";
}
if ($cat !== '') {
  // con utf8mb4_unicode_ci ya es case/accent-insensitive
  $where[] = "category = ?";
  $params[] = $cat; $types .= 's';
}
if ($q !== '') {
  $where[] = "(name LIKE ? COLLATE utf8mb4_unicode_ci OR `desc` LIKE ? COLLATE utf8mb4_unicode_ci)";
  $needle = "%$q%";
  array_push($params, $needle, $needle);
  $types .= 'ss';
}

// ================== SQL ==================
$sql = "SELECT
          id,
          category,
          name,
          `desc`   AS description,
          price,
          img,
          is_active
        FROM productos";
if ($where) $sql .= ' WHERE ' . implode(' AND ', $where);
$sql .= ' ORDER BY category, name';

try {
  if ($params) {
    $st = $mysqli->prepare($sql);
    $st->bind_param($types, ...$params);
    $st->execute();
    $res = $st->get_result();
  } else {
    $res = $mysqli->query($sql);
  }

  // Tasa opcional
  $tasa = 1.0;
  if ($rt = $mysqli->query("SELECT tasa FROM tasa_bcv ORDER BY id DESC LIMIT 1")) {
    if ($t = $rt->fetch_assoc()) $tasa = (float)$t['tasa'];
  }

  $out = [];
  $missing = 0;
  while ($row = $res->fetch_assoc()) {
    $imgResolved = resolve_img_ci((string)$row['img']);
    if ($imgResolved === '/imgproductos/placeholder.png') $missing++;
    $out[] = [
      'id'          => (int)$row['id'],
      'category'    => (string)$row['category'],
      'name'        => (string)$row['name'],
      'description' => (string)$row['description'],
      'price'       => (float)$row['price'],
      'price_$'    => round(((float)$row['price']) * $tasa, 2),
      'img'         => $imgResolved,
      'is_active'   => (int)$row['is_active'] ? 1 : 0,
      'available'   => (int)$row['is_active'] ? 1 : 0
    ];
  }

  if ($debug) {
    header('X-Debug-Table: productos');
    header('X-Debug-Count: '.count($out));
    header('X-Debug-Missing-Images: '.$missing);
  }

  echo json_encode($out, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'error'=>'DB_ERROR','detail'=>$e->getMessage(),'sql'=>$sql]);
}
